// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import de.caff.util.GlobMatcher;
import de.caff.util.debug.Debug;

import javax.swing.*;
import javax.swing.border.Border;
import java.awt.*;
import java.lang.reflect.Array;
import java.util.*;
import java.util.List;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class UIResourceCollection
{
  /** Resource types for which warning message where already given. */
  private static final Set<String> warnedResources = new HashSet<>();

  /** Mapping of strings to keys. Stupid UIDefaults uses various keys, not only strings. */
  private final Map<String, Object> idsToKeys = new HashMap<>();
  /** Mapping of stringized keys to resource handlers. */
  private final Map<String, UIResourceHandler> resourceHandlers = new HashMap<>();
  /** The UI defaults wrapped in this collection. */
  private final UIDefaults defaults;

  /**
   *  Creator of a UI resource handler.
   */
  private static interface UIResourceHandlerCreator
  {
    /**
     *  Get a resource handler.
     *  @param id the id of the handler
     *  @return the new handler
     */
    UIResourceHandler createHandler(String id);
  }

  private static class ArrayUIResourceHandlerCreator
          implements UIResourceHandlerCreator
  {
    private final UIResourceHandlerCreator subCreator;

    private ArrayUIResourceHandlerCreator(UIResourceHandlerCreator subCreator)
    {
      this.subCreator = subCreator;
    }

    /**
     * Get a resoure handler.
     *
     * @param id the id of the handler
     * @return the new handler
     */
    @Override
    public UIResourceHandler createHandler(String id)
    {
      return new ArrayUIResourceHandler(id,
                                        subCreator.createHandler("[]"));
    }
  }

  private static class ListUIResourceHandlerCreator
          implements UIResourceHandlerCreator
  {
    private final UIResourceHandlerCreator subCreator;

    private ListUIResourceHandlerCreator(UIResourceHandlerCreator subCreator)
    {
      this.subCreator = subCreator;
    }

    /**
     * Get a resoure handler.
     *
     * @param id the id of the handler
     * @return the new handler
     */
    @Override
    public UIResourceHandler createHandler(String id)
    {
      return new ListUIResourceHandler(id,
                                       subCreator.createHandler("[]"));
    }
  }

  private static final UIResourceHandlerCreator GRADIENT_UI_RESOURCE_HANDLER_CREATOR =
          GradientUIResourceHandler::new;

  /** Mapping of classes to handler creators. */
  private static final Map<Class<?>, UIResourceHandlerCreator> CREATOR_MAP = new HashMap<>();
  static {
    CREATOR_MAP.put(Integer.class,
                    IntegerUIResourceHandler::new);
    CREATOR_MAP.put(Long.class,
                    LongUIResourceHandler::new);
    CREATOR_MAP.put(Color.class,
                    ColorUIResourceHandler::new);
    CREATOR_MAP.put(Class.class,
                    ClassUIResourceHandler::new);
    CREATOR_MAP.put(java.lang.reflect.Method.class,
                    MethodUIResourceHandler::new);
    CREATOR_MAP.put(Border.class,
                    BorderUIResourceHandler::new);
    CREATOR_MAP.put(String.class,
                    StringUIResourceHandler::new);
    CREATOR_MAP.put(Boolean.class,
                    BooleanUIResourceHandler::new);
    CREATOR_MAP.put(Character.class,
                    CharacterUIResourceHandler::new);
    CREATOR_MAP.put(java.awt.Font.class,
                    FontUIResourceHandler::new);
    CREATOR_MAP.put(java.awt.Insets.class,
                    InsetsUIResourceHandler::new);
    CREATOR_MAP.put(java.awt.Dimension.class,
                    DimensionUIResourceHandler::new);
    CREATOR_MAP.put(Float.class,
                    FloatUIResourceHandler::new);
  }

  /**
   *  Default constructor.
   *  This uses the UI defaults from the UI manager.
   */
  public UIResourceCollection()
  {
    this(UIManager.getDefaults());
  }

  /**
   *  Get a resource handler creator for a given class.
   *  @param clazz class to look for
   *  @return matching creator or {@code null} if no creator is defined for this class
   */
  private static UIResourceHandlerCreator getResourceHandlerCreator(Class<?> clazz)
  {
    if (CREATOR_MAP.containsKey(clazz)) {
      return CREATOR_MAP.get(clazz);
    }
    // check interfaces
    Class<?>[] interfaces = clazz.getInterfaces();
    for (int i = interfaces.length - 1;  i >= 0;  --i) {
      UIResourceHandlerCreator creator = getResourceHandlerCreator(interfaces[i]);
      if (creator != null) {
        return creator;
      }
    }
    // check super classes and their interfaces
    Class<?> superClazz = clazz.getSuperclass();
    if (superClazz != null) {
      UIResourceHandlerCreator creator = getResourceHandlerCreator(superClazz);
      if (creator != null) {
        return creator;
      }
    }
    return null;
  }

  private static Class<?> getCommonArrayElementClass(Object array)
  {
    Class<?> common = null;
    int length = Array.getLength(array);
    for (int i = 0;  i < length;  ++i) {
      Object elem = Array.get(array, i);
      if (elem != null) {
        if (common == null) {
          common = elem.getClass();
        }
        else if (common != elem.getClass()) {
          common = null;
          break;
        }
      }
    }
    return common;
  }

  private static UIResourceHandlerCreator getResourceHandlerCreator(Object obj)
  {
    if (obj == null) {
      return null;
    }
    Class<?> clazz = obj.getClass();
    if (clazz.isArray()) {
      // todo: maybe make it work for more dimensions
      Class<?> elemClass = getCommonArrayElementClass(obj);
      if (elemClass != null) {
        UIResourceHandlerCreator elemCreator = getResourceHandlerCreator(elemClass);
        if (elemCreator != null) {
          return new ArrayUIResourceHandlerCreator(elemCreator);
        }
      }
      return null;
    }
    else if (obj instanceof List) {
      // todo: maybe make it work for more dimensions
      Class<?> elemClass = getCommonArrayElementClass(((List)obj).toArray());
      if (elemClass != null) {
        UIResourceHandlerCreator elemCreator = getResourceHandlerCreator(elemClass);
        if (elemCreator != null) {
          return new ListUIResourceHandlerCreator(elemCreator);
        }
      }
      else {
        // gradient?
        List<?> list = (List<?>)obj;
        if (list.size() == GradientUIResourceHandler.HELPER.length) {
          boolean okay = true;
          for (int i = 0;  i < GradientUIResourceHandler.HELPER.length;  ++i) {
            Object elem = list.get(i);
            if (elem == null  ||
                !isAssignable(i, elem)) {
              okay = false;
              break;
            }
          }
          if (okay) {
            return GRADIENT_UI_RESOURCE_HANDLER_CREATOR;
          }
        }
      }
      return null;
    }
    else {
      return getResourceHandlerCreator(clazz);
    }
  }

  /**
   * Check whether an element is assignable to a helper.
   * This method exists only for its annotation.
   * @param i helper index
   * @param elem element
   * @return {@code true} if the element is assignable,<br>
   *         {@code false} if the element is not assignable
   */
  @SuppressWarnings("unchecked")
  private static boolean isAssignable(int i, Object elem)
  {
    return GradientUIResourceHandler.HELPER[i].getHandledClass().isAssignableFrom(elem.getClass());
  }

  /**
   *  Create a resource handler.
   *  @param id     the id of the handler
   *  @param object the default object
   *  @return new handler or {@code null} if no creator is defined for the class
   */
  private static UIResourceHandler createResourceHandler(String id, Object object)
  {
    UIResourceHandlerCreator creator = getResourceHandlerCreator(object);
    return creator != null ? creator.createHandler(id) : null;
  }

  /**
   *  Constructor.
   *  @param defaults UI defaults
   */
  public UIResourceCollection(UIDefaults defaults)
  {
    this.defaults = defaults;
    for (Enumeration<?> iterator = defaults.keys(); iterator.hasMoreElements();) {
      Object key   = iterator.nextElement();
      Object value = defaults.get(key);
      String id = key.toString();
      if (value != null) {
        idsToKeys.put(id, key);
        UIResourceHandler handler = createResourceHandler(id, value);
        if (handler != null) {
          resourceHandlers.put(id, handler);
        }
        else {
          String className = value.getClass().toString();
          if (!warnedResources.contains(className)) {
            warnedResources.add(className);
            Debug.warn("No handler for %0 (id: %1)", className, id);
          }
        }
      }
      else {
        Debug.warn("Cannot determine class for UI resource values with key "+key+": null value");
      }
    }
  }

  /**
   *  Set a value in the wrapped UI defaults.
   *  @param id    the id of the value
   *  @param value the value in textual form
   *  @return {@code true} if the value was set,<br>
   *          {@code false} if the value was not set because there was no handler
   *                             or the textual value was not convertable
   */
  public boolean setValue(String id, String value)
  {
    UIResourceHandler handler = resourceHandlers.get(id);
    if (handler != null) {
      return handler.setValue(this, value);
    }
    Debug.warn("No resource handler for id "+id);
    return false;
  }

  /**
   *  Set some values in the wrapped UI defaults.
   *  @param glob  glob mask with <tt>*</tt> (any number of chars), <tt>?</tt> (one char), or
   *               <tt>[ABX-Z]</tt> (a char out of a set)
   *  @param value value to set with matching keys
   *  @return number of keys where setting succeded
   *  @see de.caff.util.GlobMatcher
   */
  public int setValues(String glob, String value)
  {
    int result = 0;
    GlobMatcher matcher = new GlobMatcher(glob);
    // NOTE: Enumeration needed because Java 1.4 fails to support
    //       iterator interfaces correctly!
    for (String key : resourceHandlers.keySet()) {
      if (matcher.isMatching(key)) {
        if (setValue(key, value)) {
          Debug.trace("Matching key '%0' set to '%1'", key, value);
          ++result;
        }
        else {
          Debug.warn("Invalid value '%1' for key '%0'", key, value);
        }
      }
    }
    return result;
  }

  /**
   *  Is there a handler for a given id?
   *  @param id the resource id
   *  @return the answer
   */
  public boolean hasHandler(String id)
  {
    return resourceHandlers.containsKey(id);
  }

  /**
   *  Set the value in the UI defaults.
   *  This is called by resource handlers which know only about IDs,
   *  but for some dark reason keys in the UIDefaults can be arbitrary objects.
   *  @param id    the stringized key
   *  @param value the value to set
   *  @return {@code true} if the id could be resolved into a key,<br>
   *          {@code false} otherwise
   */
  boolean setUIValue(String id, Object value)
  {
    Object key = idsToKeys.get(id);
    if (key == null) {
      Debug.warn("No key for id %0", id);
      return false;
    }
    else {
      defaults.put(key, value);
      return true;
    }
  }

  /**
   *  Test code.
   *  @param args ignored
   */
  public static void main(String[] args)
  {
    UIDefaults defaults = UIManager.getDefaults();
    Set<Object> sorted = new TreeSet<>((o1, o2) -> {
      if (o1 == null) {
        return o2 == null ? 0 : -1;
      }
      else if (o2 == null) {
        return 1;
      }
      else {
        return o1.toString().compareTo(o2.toString());
      }
    });
    // NOTE: Enumeration needed because Java 1.4 fails to support
    //       iterator interfaces correctly!
    for (Enumeration<?> iterator = defaults.keys(); iterator.hasMoreElements();) {
      Object key = iterator.nextElement();
      sorted.add(key);
    }
    UIResourceCollection uires = new UIResourceCollection(defaults);
    System.out.println("<table>");
    System.out.println("<tr><th>Key</th><th>Class</th><th>Default Value</th></tr>");
    for (Object key : sorted) {
      Object value = defaults.get(key);
      if (uires.hasHandler(key.toString())) {
        System.out.print("<tr class=\"handled\">");
      }
      else {
        System.out.print("<tr class=\"unhandled\">");
      }
      System.out.print("<td>");
      System.out.print(key);
      System.out.print("</td>");
      System.out.print("<td>");
      outputClass(value);
      System.out.print("</td>");
      System.out.print("<td>");
      outputValue(value);
      System.out.print("</td>");
      System.out.println("</tr>");
    }
    System.out.println("</table>");
  }

  private static void outputValue(Object obj)
  {
    if (obj == null) {
      System.out.print("<null>");
    }
    else {
      if (obj.getClass().isArray()) {
        int length = Array.getLength(obj);
        System.out.print("[");
        System.out.print(length);
        System.out.print("]={");
        for (int i = 0;  i < length;  ++i) {
          if (i > 0) {
            System.out.print(", ");
          }
          outputValue(Array.get(obj, i));
        }
        System.out.print("}");
      }
      else if (obj instanceof List) {
        List<?> list = (List<?>)obj;
        int length = list.size();
        System.out.print("["+ length + "]={");
        boolean first = true;
        for (Object o : list) {
          if (first) {
            first = false;
          }
          else {
            System.out.print(", ");
          }
          outputValue(o);
        }
        System.out.print("}");
      }
      else {
        System.out.print(obj);
      }
    }
  }

  private static void outputClass(Object obj)
  {
    if (obj == null) {
      System.out.print("???");
    }
    else {
      Class<?> aClass = obj.getClass();
      if (aClass.isArray()) {
        int length = Array.getLength(obj);
        System.out.print("Array["+ length +"]={");
        for (int i = 0;  i < length;  ++i) {
          if (i > 0) {
            System.out.print(", ");
          }
          outputClass(Array.get(obj, i));
        }
        System.out.print("}");
      }
      else if (obj instanceof List) {
        List<?> list = (List<?>)obj;
        int length = list.size();
        System.out.print("List["+ length + "]={");
        boolean first = true;
        for (Object o : list) {
          if (first) {
            first = false;
          }
          else {
            System.out.print(", ");
          }
          outputClass(o);
        }
        System.out.print("}");
      }
      else {
        System.out.print(aClass.getName());
      }
    }
  }

}
