// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import de.caff.util.debug.Debug;

import java.lang.reflect.Method;
import java.util.regex.Pattern;

/**
 *  UI resource handler for class values.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class MethodUIResourceHandler
        extends UIResourceHandler
{
  private static final Pattern SEPARATOR_PATTERN = Pattern.compile(",");

  /**
   *  Constructor.
   *  @param id ID of resource
   */
  public MethodUIResourceHandler(String id)
  {
    super(id);
  }

  /**
   * Convert a textual representation of a value to a value.
   *
   * @param text textual representation
   * @return the value or {@code null} if the text cannot be converted
   * @see #fromValue(Object)
   */
  @Override
  protected Object toValue(String text)
  {
    int openParen = text.indexOf('(');
    int closeParen = text.indexOf(')');
    if (openParen > 0  &&  closeParen > openParen) {
      int blank = text.lastIndexOf(' ', openParen);
      if (blank > 0) {
        int dot = text.lastIndexOf('.', openParen);
        if (dot > blank) {
          try {
            Class<?> clazz = Class.forName(text.substring(blank+1, dot));
            Class<?>[] parameters;
            if (openParen + 1 == closeParen) {
              parameters = new Class<?>[0];
            }
            else {
              String[] params = SEPARATOR_PATTERN.split(text.substring(openParen + 1, closeParen));
              parameters = new Class<?>[params.length];
              for (int p = params.length - 1;  p >= 0;  --p) {
                parameters[p] = Class.forName(params[p]);
              }
            }
            return clazz.getMethod(text.substring(dot + 1, openParen), parameters);
          } catch (ClassNotFoundException e) {
            Debug.error(e);
          } catch (NoSuchMethodException e) {
            Debug.error(e);
          }
        }
      }
    }
    return null;
  }

  /**
   * Get the string representation of a value.
   *
   * @param value the UI value
   * @return the value in textual form
   * @see #toValue(String)
   */
  @Override
  protected String fromValue(Object value)
  {
    if (value != null) {
      try {
        Method method = (Method)value;
        StringBuilder b = new StringBuilder(method.getName());
        b.append("(");
        Class<?>[] params = method.getParameterTypes(); // avoid clone
        for (int p = 0; p < params.length; p++) {
          b.append(getTypeName(params[p]));
          if (p < (params.length - 1)) {
            b.append(",");
          }
        }
        b.append(")");
        return b.toString();
      } catch (Exception e) {
        Debug.error(e);
      }
    }
    return super.fromValue(value);
  }

  /**
   * Get the class handled by this resource handler.
   *
   * @return handled class
   */
  @Override
  public Class<?> getHandledClass()
  {
    return Method.class;
  }

  // borrowed from java.lang.reflect.Field
  private static String getTypeName(Class<?> type) {
    if (type.isArray()) {
      try {
        Class<?> cl = type;
        int dimensions = 0;
        while (cl.isArray()) {
          dimensions++;
          cl = cl.getComponentType();
        }
        StringBuilder sb = new StringBuilder();
        sb.append(cl.getName());
        for (int i = 0; i < dimensions; i++) {
          sb.append("[]");
        }
        return sb.toString();
      } catch (Throwable e) { /*FALLTHRU*/ }
    }
    return type.getName();
  }
}
