// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import java.util.ArrayList;
import java.util.List;

/**
 *  UI Resource handler for arrays of other resources.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ListUIResourceHandler
        extends ArrayUIResourceHandler
{

  /**
   *  Constructor.
   *  @param id ID of resource
   *  @param elementHandler handler for the array elements
   */
  public ListUIResourceHandler(String id, UIResourceHandler elementHandler)
  {
    super(id, elementHandler);
  }

  /**
   * Convert a textual representation of a value to a value.
   *
   * @param text textual representation
   * @return the value or {@code null} if the text cannot be converted
   */
  @Override
  protected Object toValue(String text)
  {
    List<String> parts = split(text);
    if (parts != null) {
      List<Object> list = new ArrayList<>(parts.size());
      for (String part: parts) {
        Object value = elementHandler.toValue(part);
        if (value == null) {
          return null;
        }
        list.add(value);
      }
      return list;
    }
    return null;
  }

  /**
   * Get the string representation of a value.
   *
   * @param value the UI value
   * @return the value in textual form
   * @see #toValue(String)
   */
  @Override
  protected String fromValue(Object value)
  {
    if (value != null) {
      List<?> list = (List<?>)value;
      StringBuilder b = new StringBuilder(ARRAY_START);
      boolean first = true;
      for (Object obj : list) {
        if (first) {
          first = false;
        }
        else {
          b.append(ELEMENT_SEPARATOR);
        }
        b.append(elementHandler.fromValue(obj));
      }
      b.append(ARRAY_END);
      return b.toString();
    }
    return null;
  }
}
