// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import de.caff.util.debug.Debug;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 *  UI Resource handler for gradients.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class GradientUIResourceHandler
        extends UIResourceHandler
{
  private static final FloatUIResourceHandler FLOAT_HELPER = new FloatUIResourceHandler("foo");
  private static final ColorUIResourceHandler COLOR_HELPER = new ColorUIResourceHandler("bar");
  static final UIResourceHandler[] HELPER = {
          FLOAT_HELPER,
          COLOR_HELPER,
          COLOR_HELPER
  };
  private static final Pattern PATTERN_COMMA = Pattern.compile(",");

  /**
   *  Constructor.
   *  @param id ID of resource
   */
  public GradientUIResourceHandler(String id)
  {
    super(id);
  }

  /**
   * Convert a textual representation of a value to a value.
   *
   * @param text textual representation
   * @return the value or {@code null} if the text cannot be converted
   * @see #fromValue(Object)
   */
  @Override
  protected Object toValue(String text)
  {
    String[] parts = PATTERN_COMMA.split(text);
    if (parts.length == HELPER.length) {
      List<Object> result = new ArrayList<>(parts.length);
      for (int p = 0;  p < parts.length;  ++p) {
        Object value = HELPER[p].toValue(parts[p]);
        if (value == null) {
          return null;
        }
        result.add(value);
      }
      return result;
    }
    Debug.error("Not %0 elements in gradient definition: %1",
                HELPER.length,
                text);
    return null;
  }

  /**
   * Get the string representation of a value.
   *
   * @param value the UI value
   * @return the value in textual form
   * @see #toValue(String)
   */
  @Override
  protected String fromValue(Object value)
  {
    if (value != null) {
      try {
        List<?> list = (List<?>)value;
        if (list.size() == HELPER.length) {
          StringBuilder b = new StringBuilder();
          for (int i = 0;  i < HELPER.length;  ++i) {
            String part = HELPER[i].fromValue(list.get(i));
            if (part == null) {
              return null;
            }
            if (i > 0) {
              b.append(",");
            }
            b.append(part);
          }
          return b.toString();
        }
      } catch (ClassCastException x) {
        Debug.error(x);
      }
    }
    return null;
  }

  /**
   * Get the class handled by this resource handler.
   *
   * @return handled class
   */
  @Override
  public Class<?> getHandledClass()
  {
    return List.class;
  }
}
