// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import de.caff.util.debug.Debug;

/**
 *  UI resource handler for class values.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ClassUIResourceHandler
        extends UIResourceHandler
{
  /**
   *  Constructor.
   *  @param id ID of resource
   */
  public ClassUIResourceHandler(String id)
  {
    super(id);
  }

  /**
   * Convert a textual representation of a value to a value.
   *
   * @param text textual representation
   * @return the value or {@code null} if the text cannot be converted
   * @see #fromValue(Object)
   */
  @Override
  protected Object toValue(String text)
  {
    try {
      return Class.forName(text);
    } catch (Throwable e) {
      Debug.error(e);
    }
    return null;
  }

  /**
   * Get the string representation of a value.
   *
   * @param value the UI value
   * @return the value in textual form
   * @see #toValue(String)
   */
  @Override
  protected String fromValue(Object value)
  {
    if (value != null) {
      try {
        Class<?> clazz = (Class)value;
        return clazz.getName();
      } catch (ClassCastException x) {
        Debug.error(x);
      }
    }
    return super.fromValue(value);
  }

  /**
   * Get the class handled by this resource handler.
   *
   * @return handled class
   */
  @Override
  public Class<?> getHandledClass()
  {
    return Class.class;
  }
}
