// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import de.caff.generics.Empty;
import de.caff.i18n.I18n;
import de.caff.util.ColorDecoder;
import de.caff.util.debug.Debug;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.plaf.BorderUIResource;
import java.awt.*;
import java.text.ParseException;
import java.util.Collection;
import java.util.LinkedList;
import java.util.MissingResourceException;

/**
 *  UI Resource handler for border values.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class BorderUIResourceHandler
        extends UIResourceHandler
{
  /**
   *  Constructor.
   *  @param id ID of resource
   */
  public BorderUIResourceHandler(String id)
  {
    super(id);
  }

  /**
   * Convert a textual representation of a value to a value.
   *
   * @param text textual representation
   * @return the value or {@code null} if the text cannot be converted
   * @see #fromValue(Object)
   */
  @Override
  protected Object toValue(String text)
  {
    try {
      Border border = createBorder(text);
      return border != null ? new BorderUIResource(border) : null;
    } catch (ParseException x) {
      Debug.error(x);
    }
    return null;
  }

  /**
   * Get the string representation of a value.
   *
   * @param value the UI value
   * @return the value in textual form
   * @see #toValue(String)
   */
  @Override
  protected String fromValue(Object value)
  {
    try {
      Border res = (Border)value;
      if (res != null) {
        // todo: make this correct so we can parse it
        return res.toString();
      }
    } catch (ClassCastException x) {
      Debug.error(x);
    }
    return null;
  }

  /**
   *  Get the arguments from a comma-separated list of arguments,
   *
   *  @param str comma-separated list encapsuled in parenthesises
   *  @return array with arguments
   *  @throws ParseException on format errors
   */
  private static String[] getArguments(String str) throws ParseException
  {
    str = str.trim();
    if (str.charAt(0) != '(') {
      throw new ParseException("Not a valid argument list: "+str,
                               0);
    }
    if (str.charAt(str.length() - 1) != ')') {
      throw new ParseException("Not a valid argument list: "+str,
                               str.length() - 1);
    }
    str = str.substring(1, str.length() - 1);
    // now split at ',' while keeping strings and parenthesises
    Collection<String> result = new LinkedList<>();
    if (!str.isEmpty()) {
      str += ',';  // adding an argument separator here simpilfies the logic of the following loop
      boolean inString = false;
      int openParen = 0;
      int lastStart = 0;
      for (int c = 0;  c < str.length();  ++c) {
        char ch = str.charAt(c);
        if (inString) {
          if (ch == '\'') {
            inString = false;
          }
        }
        else {
          switch (ch) {
          case '\'':
            inString = true;
            break;

          case '(':
            ++openParen;
            break;

          case ')':
            --openParen;
            if (openParen < 0) {
              throw new ParseException("Too many closing parenthesis in "+str.substring(0, str.length() - 1), c);
            }
            break;

          case ',':
            if (openParen == 0) {
              String part = str.substring(lastStart, c).trim();
              if (part.length() >= 2  &&
                  part.charAt(0) == '\''  &&
                  part.charAt(part.length()-1) == '\'') {
                part = part.substring(1, part.length() - 1);
              }
              result.add(part);
              lastStart = c + 1;
            }
            break;
          }
        }
      }
      if (openParen != 0) {
        throw new ParseException("Unbalanced parenthesis in "+str.substring(0, str.length() - 1),
                                 str.length() - 1);
      }
      if (inString) {
        throw new ParseException("Unbalanced ' in "+str.substring(0, str.length() - 1),
                                 str.length() - 1);
      }
    }
    return result.toArray(Empty.STRING_ARRAY);
  }

  /**
   *  Get an integer value from a string,
   *  @param descr string containing integer
   *  @return integer value
   *  @throws ParseException if string does not contain an integer
   */
  private static int getInt(String descr) throws ParseException
  {
    try {
      return Integer.parseInt(descr);
    } catch (NumberFormatException e) {
      throw new ParseException("Not an integer number: "+descr,
                               descr.length() - 1);
    }
  }

  /**
   *  Create a color from a color description.
   *  @param descr color description  (see {@link de.caff.util.ColorDecoder#decode(String)})
   *  @return decoded color
   *  @throws ParseException if color descripion is no valid
   */
  private static Color parseColor(String descr) throws ParseException
  {
    Color ret = ColorDecoder.decode(descr);
    if (ret == null) {
      throw new ParseException("Cannot decode color value: "+descr,
                               0);
    }
    return ret;
  }

  private static String resolveI18n(String tag)
  {
    if (tag == null) {
      return null;
    }
    if (tag.startsWith("${")  &&  tag.endsWith("}")) {
      try {
        return I18n.getString(tag.substring(2, tag.length() - 1));
      } catch (MissingResourceException e) {
        Debug.error("Cannot resolve tag '"+tag+"'");
        return tag.substring(2, tag.length() - 1);  // this is the best guess
      }
    }
    return tag;
  }

  public static Border createBorder(String descr) throws ParseException
  {
    String ldescr = descr.toLowerCase();
    if (ldescr.startsWith("bevel(")) {
      String[] parts = getArguments(descr.substring(5));
      int bevelType;
      if ("lowered".equalsIgnoreCase(parts[0])) {
        bevelType = BevelBorder.LOWERED;
      }
      else if ("raised".equalsIgnoreCase(parts[0])) {
        bevelType = BevelBorder.RAISED;
      }
      else {
        throw new ParseException("Not a valid bevel type: "+parts[0],
                                 5);
      }
      switch(parts.length) {
      case 1:
        return BorderFactory.createBevelBorder(bevelType);
      case 3:
        return BorderFactory.createBevelBorder(bevelType,
                                               parseColor(parts[1]),
                                               parseColor(parts[2]));
      case 5:
        return BorderFactory.createBevelBorder(bevelType,
                                               parseColor(parts[1]),
                                               parseColor(parts[2]),
                                               parseColor(parts[3]),
                                               parseColor(parts[4]));
      default:
        throw new ParseException("Not a valid number of color parameters for bevel border: "+descr,
                                 5);
      }
    }
    else if (ldescr.startsWith("compound(")) {
      String[] parts = getArguments(descr.substring(8));
      switch (parts.length) {
      case 0:
        // no very useful
        return BorderFactory.createCompoundBorder();
      case 2:
        return BorderFactory.createCompoundBorder(createBorder(parts[0]),
                                                  createBorder(parts[1]));
      default:
        throw new ParseException("Not a valid number of parameters for compound border: "+descr,
                                 8);
      }
    }
    else if (ldescr.startsWith("empty(")) {
      String[] parts = getArguments(descr.substring(5));
      switch (parts.length) {
      case 0:
        return BorderFactory.createEmptyBorder();
      case 4:
        return BorderFactory.createEmptyBorder(getInt(parts[0]),
                                               getInt(parts[1]),
                                               getInt(parts[2]),
                                               getInt(parts[3]));
      default:
        throw new ParseException("Not a valid number of parameters for empty border: "+descr,
                                 5);
      }
    }
    else if (ldescr.startsWith("etched(")) {
      String[] parts = getArguments(descr.substring(6));
      if (parts.length == 0) {
        return BorderFactory.createEtchedBorder();
      }
      if (parts.length == 2) {
        return BorderFactory.createEtchedBorder(parseColor(parts[0]),
                                                parseColor(parts[1]));
      }
      int etchType;
      if ("lowered".equalsIgnoreCase(parts[0])) {
        etchType = EtchedBorder.LOWERED;
      }
      else if ("raised".equalsIgnoreCase(parts[0])) {
        etchType = EtchedBorder.RAISED;
      }
      else {
        throw new ParseException("Not a valid etched type: "+parts[0],
                                 6);
      }
      switch (parts.length) {
      case 1:
        return BorderFactory.createEtchedBorder(etchType);
      case 3:
        return BorderFactory.createEtchedBorder(etchType,
                                                parseColor(parts[0]),
                                                parseColor(parts[1]));
      default:
        throw new ParseException("Not a valid number of parameters for etched border: "+descr,
                                 6);
      }
    }
    else if (ldescr.startsWith("line(")) {
      String[] parts = getArguments(descr.substring(4));
      switch (parts.length) {
      case 1:
        return BorderFactory.createLineBorder(parseColor(parts[0]));
      case 2:
        return BorderFactory.createLineBorder(parseColor(parts[0]),
                                              getInt(parts[1]));
      default:
        throw new ParseException("Not a valid number of parameters for empty border: "+descr,
                                 4);
      }
    }
    else if (ldescr.startsWith("loweredbevel(")) {
      String[] parts = getArguments(descr.substring(12));
      switch(parts.length) {
      case 0:
        return BorderFactory.createLoweredBevelBorder();
      default:
        throw new ParseException("Not a valid number of parameters for loweredBevel border: "+descr,
                                 12);
      }
    }
    else if (ldescr.startsWith("matte(")) {
      String[] parts = getArguments(descr.substring(5));
      switch(parts.length) {
      case 5:
        return BorderFactory.createMatteBorder(getInt(parts[0]),
                                               getInt(parts[1]),
                                               getInt(parts[2]),
                                               getInt(parts[3]),
                                               parseColor(parts[4]));
      default:
        throw new ParseException("Not a valid number of parameters for matte border: "+descr,
                                 5);
      }
    }
    else if (ldescr.startsWith("raisedbevel(")) {
      String[] parts = getArguments(descr.substring(11));
      switch(parts.length) {
      case 0:
        return BorderFactory.createRaisedBevelBorder();
      default:
        throw new ParseException("Not a valid number of parameters for raisedBevel border: "+descr,
                                 11);
      }
    }
    else if (ldescr.startsWith("titled(")) {
      String[] parts = getArguments(descr.substring(6));
      switch(parts.length) {
      case 1:
        return BorderFactory.createTitledBorder(resolveI18n(parts[0]));
      case 2:
        return BorderFactory.createTitledBorder(createBorder(parts[0]),
                                                resolveI18n(parts[1]));
      case 4:
        int justify;
        if ("left".equalsIgnoreCase(parts[2])) {
          justify = TitledBorder.LEFT;
        }
        else if ("right".equalsIgnoreCase(parts[2])) {
          justify = TitledBorder.RIGHT;
        }
        else if ("center".equalsIgnoreCase(parts[2])) {
          justify = TitledBorder.CENTER;
        }
        else if ("leading".equalsIgnoreCase(parts[2])) {
          justify = TitledBorder.LEADING;
        }
        else if ("trailing".equalsIgnoreCase(parts[2])) {
          justify = TitledBorder.TRAILING;
        }
        else if ("default".equalsIgnoreCase(parts[2])) {
          justify = TitledBorder.DEFAULT_JUSTIFICATION;
        }
        else {
          throw new ParseException("Not a valid justification type on titled border: "+parts[2],
                                   0);
        }
        int pos;
        if ("above_top".equalsIgnoreCase(parts[3])) {
          pos = TitledBorder.ABOVE_TOP;
        }
        else if ("top".equalsIgnoreCase(parts[3])) {
          pos = TitledBorder.TOP;
        }
        else if ("below_top".equalsIgnoreCase(parts[3])) {
          pos = TitledBorder.BELOW_TOP;
        }
        else if ("above_bottom".equalsIgnoreCase(parts[3])) {
          pos = TitledBorder.ABOVE_BOTTOM;
        }
        else if ("bottom".equalsIgnoreCase(parts[3])) {
          pos = TitledBorder.BOTTOM;
        }
        else if ("below_bottom".equalsIgnoreCase(parts[3])) {
          pos = TitledBorder.BELOW_BOTTOM;
        }
        else if ("default".equalsIgnoreCase(parts[3])) {
          pos = TitledBorder.DEFAULT_POSITION;
        }
        else {
          throw new ParseException("Not a valid position type on titled border: "+parts[3],
                                   0);
        }

        return BorderFactory.createTitledBorder(createBorder(parts[0]),
                                                resolveI18n(parts[1]),
                                                justify,
                                                pos);
      default:
        throw new ParseException("Not a valid number of parameters for titled border: "+descr,
                                 6);
      }
    }
    else {
      throw new ParseException("Unknown border type: "+descr, 0);
    }
  }

  /**
   * Get the class handled by this resource handler.
   *
   * @return handled class
   */
  @Override
  public Class<?> getHandledClass()
  {
    return Border.class;
  }
}
