// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2008-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.ui;

import de.caff.util.debug.Debug;

import java.util.LinkedList;
import java.util.List;

/**
 *  UI Resource handler for arrays of other resources.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ArrayUIResourceHandler
        extends UIResourceHandler
{
  protected static final char   MARKER            = '@';
  protected static final String MARKER_STRING     = Character.toString(MARKER);
  protected static final String ARRAY_START       = MARKER_STRING + "[@";
  protected static final String ARRAY_END         = MARKER_STRING + "]@";
  protected static final String ELEMENT_SEPARATOR = MARKER_STRING + "|@";
  protected final UIResourceHandler elementHandler;

  /**
   *  Constructor.
   *  @param id ID of resource
   *  @param elementHandler handler for the array elements
   */
  public ArrayUIResourceHandler(String id, UIResourceHandler elementHandler)
  {
    super(id);
    this.elementHandler = elementHandler;
  }

  /**
   * Convert a textual representation of a value to a value.
   *
   * @param text textual representation
   * @return the value or {@code null} if the text cannot be converted
   */
  @Override
  protected Object toValue(String text)
  {
    List<String> parts = split(text);
    if (parts != null) {
      Object[] array = new Object[parts.size()];
      int index = 0;
      for (String part : parts) {
        array[index] = elementHandler.toValue(part);
        if (array[index] == null) {
          return null;
        }
        ++index;
      }
      return array;
    }
    return null;
  }

  protected static List<String> split(String value)
  {
    List<String> result = new LinkedList<>();
    value = value.trim();
    if (!value.startsWith(ARRAY_START)) {
      Debug.error("Array not starting with %0", ARRAY_START);
      return null;
    }
    if (!value.endsWith(ARRAY_END)) {
      Debug.error("Array not ending with %0", ARRAY_END);
      return null;
    }
    String inner = value.substring(ARRAY_START.length(),
                                   value.length() - ARRAY_END.length());
    StringBuilder b = new StringBuilder();
    int pos = 0;
    int openCount = 0;
    int length = inner.length();
    while (pos < length) {
      char ch = inner.charAt(pos);
      if (ch == MARKER) {
        String sub = inner.substring(pos);
        if (openCount == 0  &&
            sub.startsWith(ELEMENT_SEPARATOR)) {
          result.add(b.toString());
          b.setLength(0);
          pos += ELEMENT_SEPARATOR.length();
        }
        else if (sub.startsWith(ARRAY_START)) {
          ++openCount;
          b.append(ARRAY_START);
          pos += ARRAY_START.length();
        }
        else if (sub.startsWith(ARRAY_END)) {
          if (--openCount < 0) {
            Debug.error("Too many closing %0 for array!", ARRAY_END);
            return null;
          }
          b.append(ARRAY_END);
          pos += ARRAY_END.length();
        }
      }
      else {
        b.append(ch);
      }
    }
    if (!result.isEmpty() ||  b.length() > 0) {
      result.add(b.toString());
    }
    return result;
  }

  /**
   * Get the string representation of a value.
   *
   * @param value the UI value
   * @return the value in textual form
   * @see #toValue(String)
   */
  @Override
  protected String fromValue(Object value)
  {
    if (value != null) {
      Object[] array = (Object[])value;
      StringBuilder b = new StringBuilder(ARRAY_START);
      int length = array.length;
      for (int i = 0;  i < length;  ++i) {
        if (i > 0) {
          b.append(ELEMENT_SEPARATOR);
        }
        b.append(elementHandler.fromValue(array[i]));
      }
      b.append(ARRAY_END);
      return b.toString();
    }
    return null;
  }

  /**
   * Get the class handled by this resource handler.
   *
   * @return handled class
   */
  @Override
  public Class<?> getHandledClass()
  {
    return Object[].class;
  }
}
