// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.table;

import de.caff.annotation.Nullable;

import javax.swing.table.AbstractTableModel;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * Table model displaying the properties of an object.
 * <p>
 * It considers the object's public fields and public parameters plus non-void <tt>getXXX()</tt> and
 * boolean <tt>isXXX()</tt> methods as defining properties. You can hide such a field or a method from
 * being used by marking it with the {@link PropertyHide} annotation.
 * You can change the automatically created name by marking it with the {@link PropertyName} annotation.
 * <p>
 * It uses reflection to access the properties, so it might fail in restricted environments.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ReadOnlyPropertyTableModel
        extends AbstractTableModel
{
  /** The column index of the property name. */
  public static final int COLUMN_INDEX_NAME = 0;
  /** The column index of the property value. */
  public static final int COLUMN_INDEX_VALUE = 1;

  private static final long serialVersionUID = -1212470710766192947L;

  private final List<PropertyResolver> resolvers;
  private final Object object;

  /**
   * Constructor.
   * @param object object to handle
   */
  public ReadOnlyPropertyTableModel(@Nullable Object object)
  {
    this.object = object;
    final List<PropertyResolver> collect = new LinkedList<>();
    if (object != null) {
      Class<?> klasse = object.getClass();
      for (Field f : klasse.getFields()) {
        if (Modifier.isStatic(f.getModifiers())) {
          continue;
        }
        if (FieldBasedPropertyResolver.isProperty(f)) {
          collect.add(new FieldBasedPropertyResolver(f));
        }
      }
      for (Method m : klasse.getMethods()) {
        if (Modifier.isStatic(m.getModifiers())) {
          continue;
        }
        if (MethodBasedPropertyResolver.isProperty(m)) {
          collect.add(new MethodBasedPropertyResolver(m));
        }
      }
    }
    resolvers = new ArrayList<>(collect);
  }

  /**
   * Returns the number of rows in the model. A
   * {@code JTable} uses this method to determine how many rows it
   * should display.  This method should be quick, as it
   * is called frequently during rendering.
   *
   * @return the number of rows in the model
   * @see #getColumnCount
   */
  @Override
  public int getRowCount()
  {
    return resolvers.size();
  }

  /**
   * Returns the number of columns in the model. A
   * {@code JTable} uses this method to determine how many columns it
   * should create and display by default.
   *
   * @return the number of columns in the model
   * @see #getRowCount
   */
  @Override
  public int getColumnCount()
  {
    return 2;
  }

  /**
   * Returns the value for the cell at {@code columnIndex} and
   * {@code rowIndex}.
   *
   * @param        rowIndex        the row whose value is to be queried
   * @param        columnIndex the column whose value is to be queried
   * @return the value Object at the specified cell
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex)
  {
    PropertyResolver resolver = resolvers.get(rowIndex);
    switch (columnIndex) {
    case COLUMN_INDEX_NAME:
      return resolver.getPropertyName();

    case COLUMN_INDEX_VALUE:
      try {
        return resolver.getPropertyValue(object);
      } catch (PropertyResolveException e) {
        return e;
      }
    }
    return null;
  }

  /**
   * Returns false.
   *
   * @param rowIndex    the row being queried
   * @param columnIndex the column being queried
   * @return false
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex)
  {
    return false;
  }

  /**
   * Returns the name for a column.
   *
   * @param column the column being queried
   * @return a string containing the default name of {@code column}
   */
  @Override
  public String getColumnName(int column)
  {
    switch (column) {
    case COLUMN_INDEX_NAME:
      return "Property Name"; // todo: i18n

    case COLUMN_INDEX_VALUE:
      return "Property Value"; // todo: i18n
    }
    return null;
  }

  /**
   * Update the model.
   */
  public void update()
  {
    fireTableDataChanged();
  }
}
