// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.table;

import de.caff.annotation.NotNull;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * Property based on a method call.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class MethodBasedPropertyResolver
        extends AbstractBasicPropertyResolver
{
  static final String METHOD_PREFIX_GET = "get";
  static final String METHOD_PREFIX_IS = "is";

  /** The method used for extracting the property. */
  private final Method method;

  /**
   * Constructor.
   * @param method method to wrap
   */
  public MethodBasedPropertyResolver(Method method)
  {
    super(getPropertyNameFromMethod(method));
    this.method = method;
  }

  static String getPropertyNameFromMethod(Method m)
  {
    PropertyName propName = m.getAnnotation(PropertyName.class);
    if (propName != null) {
      return propName.value();
    }
    String name = m.getName();
    if (name.startsWith(METHOD_PREFIX_GET)) {
      return name.substring(METHOD_PREFIX_GET.length());
    }
    if (name.startsWith(METHOD_PREFIX_IS)) {
      return name.substring(METHOD_PREFIX_IS.length());
    }
    final StringBuilder sb = new StringBuilder(name.length());
    sb.append(Character.toUpperCase(name.charAt(0)));
    sb.append(name.substring(1));
    return sb.toString();
  }

  /**
   * Check whether a method defines a property value.
   * @param m method to check
   * @return {@code true} if the method defines a property<br>
   *         {@code false} if not
   */
  static boolean isProperty(@NotNull Method m)
  {
    if (!Modifier.isPublic(m.getModifiers())) {
      return false;
    }
    if (m.getParameterTypes().length > 0)  {
      return false;
    }
    final Class<?> returnType = m.getReturnType();
    if (returnType == null) {
      return false;
    }
    if (m.getAnnotation(PropertyHide.class) != null) {
      return false;
    }
    String name = m.getName();
    if (name.startsWith(METHOD_PREFIX_GET) &&
        name.length() > METHOD_PREFIX_GET.length()) {
      return true;
    }
    if (name.startsWith(METHOD_PREFIX_IS) &&
        name.length() > METHOD_PREFIX_IS.length() &&
        returnType.equals(Boolean.TYPE)) {
      return true;
    }
    return false;
  }

  /**
   * Get the value property from a given object.
   *
   * @param object the object
   * @return the value
   * @throws de.caff.util.table.PropertyResolveException
   *          on errors during the resolve
   */
  @Override
  public Object getPropertyValue(Object object) throws PropertyResolveException
  {
    try {
      return method.invoke(object);
    } catch (Throwable t) {
      throw new PropertyResolveException(t);
    }
  }
}
