// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.table;

import de.caff.annotation.NotNull;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

/**
 * Property based on an object's field.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class FieldBasedPropertyResolver
        extends AbstractBasicPropertyResolver
{
  /** The method used for extracting the property. */
  @NotNull
  private final Field field;

  /**
   * Constructor.
   * @param field field to wrap
   */
  public FieldBasedPropertyResolver(@NotNull Field field)
  {
    super(getPropertyNameFromField(field));
    this.field = field;
  }
  
  static String getPropertyNameFromField(@NotNull  Field f)
  {
    PropertyName propName = f.getAnnotation(PropertyName.class);
    if (propName != null) {
      return propName.value();
    }
    String name = f.getName();
    StringBuilder sb = new StringBuilder(name.length());
    sb.append(Character.toUpperCase(name.charAt(0)));
    sb.append(name.substring(1));
    return sb.toString();
  }

  /**
   * Check whether a field defines a property value.
   * @param f field to check
   * @return {@code true} if the field defines a property<br>
   *         {@code false} if not
   */
  static boolean isProperty(@NotNull Field f)
  {
    if (!Modifier.isPublic(f.getModifiers())) {
      return false;
    }
    if (f.getAnnotation(PropertyHide.class) != null) {
      return false;
    }
    return true;
  }

  /**
   * Get the value property from a given object.
   *
   * @param object the object
   * @return the value
   * @throws PropertyResolveException
   *          on errors during the resolve
   */
  @Override
  public Object getPropertyValue(Object object) throws PropertyResolveException
  {
    try {
      return field.get(object);
    } catch (Throwable t) {
      throw new PropertyResolveException(t);
    }
  }
}
