// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2017-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.table;

import de.caff.annotation.NotNull;

import javax.swing.*;
import javax.swing.event.TableModelEvent;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import java.awt.*;
import java.util.Vector;

/**
 * Table which can adjust its row heights.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 27, 2017
 */
public class AutoAdjustRowHeightTable
        extends JTable
{
  private static final long serialVersionUID = -954724048773141856L;
  private boolean autoAdjustingPerRow = false;

  public AutoAdjustRowHeightTable()
  {
  }

  public AutoAdjustRowHeightTable(TableModel dm)
  {
    super(dm);
  }

  public AutoAdjustRowHeightTable(TableModel dm, TableColumnModel cm)
  {
    super(dm, cm);
  }

  public AutoAdjustRowHeightTable(TableModel dm, TableColumnModel cm,
                                  ListSelectionModel sm)
  {
    super(dm, cm, sm);
  }

  public AutoAdjustRowHeightTable(int numRows, int numColumns)
  {
    super(numRows, numColumns);
  }

  public AutoAdjustRowHeightTable(Vector<? extends Vector<?>> rowData, Vector<?> columnNames)
  {
    super(rowData, columnNames);
  }

  public AutoAdjustRowHeightTable(Object[][] rowData, Object[] columnNames)
  {
    super(rowData, columnNames);
  }

  @Override
  public void setModel(@NotNull TableModel dataModel)
  {
    super.setModel(dataModel);
    adjustRowHeights();
  }

  @Override
  public void tableChanged(TableModelEvent e)
  {
    super.tableChanged(e);
    adjustRowHeights();
  }

  /**
   * Is this table auto adjusting per row?
   * @return {@code true}: if auto adjusting happens per row (see {@link #adjustRowHeightsPerRow()})<br>
   *         {@code false}: if auto adjusting happens for the complete table (see {@link #adjustRowHeightCompletely()})
   */
  public boolean isAutoAdjustingPerRow()
  {
    return autoAdjustingPerRow;
  }

  /**
   * Set whether automatic adjustment happens per row.
   * @param autoAdjustingPerRow {@code true}: to make auto adjusting happen per row (see {@link #adjustRowHeightsPerRow()})<br>
   *         {@code false}: to make auto adjusting happen for the complete table (see {@link #adjustRowHeightCompletely()})
   */
  public void setAutoAdjustingPerRow(boolean autoAdjustingPerRow)
  {
    this.autoAdjustingPerRow = autoAdjustingPerRow;
  }

  /**
   * Adjust the row heights per row.
   * That means that each row might get a different height.
   */
  public void adjustRowHeightsPerRow()
  {
    final int margin = getRowMargin();
    final int standardRowHeight = getRowHeight();
    if (standardRowHeight == 0) {
      // triggered early during creation
      return;
    }

    for (int row = getRowCount() - 1; row >= 0;  --row) {
      int rowHeight = standardRowHeight;
      for (int column = getColumnCount() - 1; column >= 0; --column) {
        Component comp = prepareRenderer(getCellRenderer(row, column), row, column);
        rowHeight = Math.max(rowHeight, comp.getPreferredSize().height);
      }
      setRowHeight(row, rowHeight + margin);
    }
  }

  /**
   * Adjust the row heights all together.
   * This means that the row height of all rows will be set
   * to fit the highest row.
   */
  public void adjustRowHeightCompletely()
  {
    int rowHeight = getRowHeight();
    if (rowHeight == 0) {
      // triggered early during creation
      return;
    }
    for (int row = getRowCount() - 1; row >= 0;  --row) {
      for (int column = getColumnCount() - 1; column >= 0; --column) {
        Component comp = prepareRenderer(getCellRenderer(row, column), row, column);
        rowHeight = Math.max(rowHeight, comp.getPreferredSize().height);
      }
    }
    setRowHeight(rowHeight);
  }

  /**
   * Adjust the row heights.
   * This triggers an adjustment depending
   * on {@link #isAutoAdjustingPerRow()}.
   */
  public void adjustRowHeights()
  {
    if (autoAdjustingPerRow) {
      adjustRowHeightsPerRow();
    }
    else {
      adjustRowHeightCompletely();
    }
  }

  @Override
  public void updateUI()
  {
    super.updateUI();
    adjustRowHeights();
  }
}
