// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2007-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.*;
import de.caff.util.Utility;
import de.caff.util.debug.Debug;
import de.caff.vic.DynamicallyScaledIcon;
import de.caff.vic.SimpleVectorImageReader;
import de.caff.vic.VectorImage;

import javax.swing.*;
import javax.swing.plaf.FontUIResource;
import javax.swing.table.JTableHeader;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.io.InputStream;
import java.net.URL;
import java.util.*;
import java.util.function.Consumer;

/**
 *  Helper class for getting things done in swing.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingHelper
{
  /** Internally reused array. */
  private static final Component[] EMPTY_COMPONENT_ARRAY = new Component[0];
  /** The default value for a minimal font size when adjusting font sizes. */
  public static final int DEFAULT_MIN_FONT_SIZE = 5;

  /** Standard icon sizes under various operating systems. */
  public static final IntIndexable STANDARD_ICON_SIZES =
          IntIndexable.viewArray(16, 24, 32, 48, 64, 72, 96, 128, 180, 256);

  /**
   * The icon loader class.
   * <p>
   * Thanks to modules loading resources is not as easy as it was before.
   * Therefore it is tried to load icons via dedicated classes from their module.
   * To define such a class prepend the resource's URL with the class name and
   * a pipe symbol ({@code |}).
   * If no such symbol is given the class provided here is used for loading.
   * It has to exported by its module by being in an opened package (generally
   * or for {@code de.caff.common.swing}.
   * <p>
   * You can set another class by setting Java property {@code default.icon.loader}
   * to the class' name. The default uses
   * {@code de.caff.gimmicks.resources.IconConstants} which contains icons used
   * in various de.caff projects.
   */
  private static final String DEFAULT_ICON_LOAD_CLASS =
          Utility.getStringParameter("default.icon.loader",
                                     "de.caff.gimmicks.resources.IconConstants");

  /**
   * Default number of text rows which are expected to be displayed with good visibility.
   * This is part of hidpi support.
   */
  public static final int DEFAULT_NUM_ROWS = 96;

  /**
   * Number of rows expressed as virtual font size.
   */
  private static final int DEFAULT_RELATIVE_FONT_SIZE = 4 * DEFAULT_NUM_ROWS / 3;

  /** Accumulated font size changes. */
  private static int accumulatedFontSizeChanges = 0;

  /**
   *  Adjust all font sizes used in the application.
   *  The given delta is just added to all font sizes in this application.
   *  This uses the default minimum size.
   *  @param plusminus      how to adjust, if positive fonts used become larger, if negative smaller
   */
  public static void adjustFontSizes(int plusminus)
  {
    adjustFontSizes(plusminus, DEFAULT_MIN_FONT_SIZE);
  }

  /**
   *  Adjust all font sizes used in the application.
   *  The given delta is just added to all font sizes in this application.
   *  @param plusminus      how to adjust, if positive fonts used become larger, if negative smaller
   *  @param allowedMinSize the minimal font size result when shrinking
   */
  public static void adjustFontSizes(int plusminus, int allowedMinSize)
  {
    adjustFontSizes(plusminus, allowedMinSize, EMPTY_COMPONENT_ARRAY);
  }

  /**
   *  Adjust all font sizes used in the application.
   *  The given delta is just added to all font sizes in this application.
   *  This uses the default minimum size.
   *  @param plusminus      how to adjust, if positive fonts used become larger, if negative smaller
   *  @param updateRoot     because for some reason the components don't update automatically,
   *                        here a root component may be added which will be explicitely updated using
   *                        the {@link #updateUI(java.awt.Component)} method
   */
  public static void adjustFontSizes(int plusminus, @Nullable Component updateRoot)
  {
    adjustFontSizes(plusminus, DEFAULT_MIN_FONT_SIZE, updateRoot);
  }

  /**
   *  Adjust all font sizes used in the application.
   *  The given delta is just added to all font sizes in this application.
   *  @param plusminus      how to adjust, if positive fonts used become larger, if negative smaller
   *  @param allowedMinSize the minimal font size result when shrinking
   *  @param updateRoot     because for some reason the components don't update automatically,
   *                        here a root component may be added which will be explicitly updated using
   *                        the {@link #updateUI(java.awt.Component)} method
   */
  public static void adjustFontSizes(int plusminus, int allowedMinSize, @Nullable Component updateRoot)
  {
    adjustFontSizes(plusminus, allowedMinSize,
                    updateRoot != null  ?
                            new Component[] { updateRoot }  :
                            EMPTY_COMPONENT_ARRAY);
  }

  /**
   *  Adjust all font sizes used in the application.
   *  The given delta is just added to all font sizes in this application.
   *  This uses the default minimum size.
   *  @param plusminus      how to adjust, if positive fonts used become larger, if negative smaller
   *  @param updateRoots    because for some reason the components don't update automatically,
   *                        here root components may be added which will be explicitly updated using
   *                        the {@link #updateUI(java.awt.Component)} method
   */
  public static void adjustFontSizes(int plusminus, Component... updateRoots)
  {
    adjustFontSizes(plusminus, DEFAULT_MIN_FONT_SIZE, updateRoots);
  }

  /**
   *  Adjust all font sizes used in the application.
   *  The given delta is just added to all font sizes in this application.
   *  @param plusminus      how to adjust, if positive fonts used become larger, if negative smaller
   *  @param allowedMinSize the minimal font size result when shrinking
   *  @param updateRoots    because for some reason the components don't update automatically,
   *                        here root components may be added which will be explicitly updated using
   *                        the {@link #updateUI(java.awt.Component)} method
   */
  public static void adjustFontSizes(int plusminus, int allowedMinSize, Component... updateRoots)
  {
    // this adapts a useful code snippet from Christian Ullenboom,
    // see http://www.tutego.com/blog/javainsel/2005/12/adjust-font-size-in-swing-applications.html
    if (plusminus == 0) {
      return;
    }
    int minSize = getMinUiFontSize();
    if (plusminus < 0) {
      if (minSize + plusminus < allowedMinSize) {
        // not make things smaller than MIN_FONT_SIZE
        if (minSize == allowedMinSize)  {
          return;
        }
        plusminus = allowedMinSize - minSize;
      }
    }
    if (plusminus == 0) {
      return;
    }
    Debug.message("Adjusting font sizes: %0", plusminus);
    accumulatedFontSizeChanges += plusminus;
    final float scaling = 1.0f + plusminus/(float)minSize;

    // the following is a workaround against occasional ConcurrentModificationExceptions
    // when calling getUniqueUIKeys() during the early program startup
    // START WORKAROUNG
    Iterable<?> uniqueUIKeys;
    int tries = 10;
    while (true) {
      try {
        uniqueUIKeys = getUniqueUIKeys();
        break;
      } catch (ConcurrentModificationException x) {
        Debug.error(x);
      }
      if (--tries <= 0) {
        Debug.error("Too many retries for setting font size! Giving up!");
        return;
      }
    }
    // END WORKAROUND

    for (Object key : uniqueUIKeys) {
      Object value = UIManager.get(key);
      //System.out.printf("%s: %s\n", key, value);

      if (value instanceof Font) {
        final Font f = (Font) value;
        final float oldSize = f.getSize2D();
        final float newSize = oldSize * scaling;
        UIManager.put(key,
                      new FontUIResource(f.deriveFont(newSize)));
      }
    }
    for (Component c : updateRoots) {
      updateUI(c);
    }
  }

  /**
   * FOr some reasons the keys of the UI defaults are sometimes duplicated.
   * This method takes care of that and returns the unique keys.
   * @return sorted iterable with unique UI keys
   */
  @NotNull
  private static Iterable<?> getUniqueUIKeys()
  {
    return Types.map(new TreeSet<>(),
                     UIManager.getDefaults().keys(),
                     String::valueOf);
  }

  /**
   * Get the minimal size used in the UI fonts.
   * @return minimal UI font size
   */
  public static int getMinUiFontSize()
  {
    int minSize = Integer.MAX_VALUE;
    for (Object key : getUniqueUIKeys()) {
      Object value = UIManager.get(key);
      //System.out.printf("%s: %s\n", key, value);
      if (value instanceof Font) {
        Font f = (Font) value;
        int size = f.getSize();
        if (size < minSize) {
          minSize = size;
        }
      }
    }
    return minSize;
  }

  /**
   *  Calls the updateUI() method on swing components recursively.
   *  @param component root component
   */
  public static void updateUI(Component component)
  {
    afterEachJComponent(component, JComponent::updateUI);
  }

  /**
   * Calls the given handler on swing components recursively.
   * The handler is called on a component before its subcomponents.
   * @param component root component
   * @param handler   consumer called for each JComponent
   */
  private static void beforeEachJComponent(@Nullable Component component,
                                           @NotNull Consumer<JComponent> handler)
  {
    forEachJComponent(component, handler, true, false);
  }

  /**
   * Calls the given handler on swing components recursively.
   * The handler is called on a component after its subcomponents.
   * @param component root component
   * @param handler    consumer called for each JComponent
   */
  private static void afterEachJComponent(@Nullable Component component,
                                           @NotNull Consumer<JComponent> handler)
  {
    forEachJComponent(component, handler, false, true);
  }

  /**
   * Calls the given handler on swing components recursively.
   * @param component root component
   * @param handler    consumer called for each JComponent
   * @param callBefore call handler before subcomponents?
   * @param callAfter  call handler after subcomponents?
   */
  private static void forEachJComponent(@Nullable Component component,
                                        @NotNull Consumer<JComponent> handler,
                                        boolean callBefore, boolean callAfter)
  {
    if (component instanceof Container) {
      final Container container = (Container)component;
      if (callBefore  &&  container instanceof JComponent) {
        handler.accept((JComponent)container);
      }
      for (int c = container.getComponentCount() - 1;  c >= 0;  --c) {
        forEachJComponent(container.getComponent(c), handler, callBefore, callAfter);
      }
      if (component instanceof JMenu) {
        final JMenu menu = (JMenu) component;
        for (int c = menu.getItemCount() - 1;  c >= 0;  --c) {
          forEachJComponent(menu.getItem(c), handler, callBefore, callAfter);
        }
      }

      if (callAfter  &&  container instanceof JComponent) {
        handler.accept((JComponent)container);
      }
    }
  }

  /**
   * Get the accumulated font size changes.
   * @return accumulated font size changes
   */
  public static int getAccumulatedFontSizeChanges()
  {
    return accumulatedFontSizeChanges;
  }

  /**
   * Enable high DPI support for the given window.
   * This goes recursively over the component tree(s) and enables high DPI support.
   * @param mainWindow application's main window
   * @param moreRoots  more root components for recursive enablement
   */
  public static void enableHiDpiSupport(@NotNull final Window mainWindow,
                                        @NotNull Component ... moreRoots)
  {
    final Component[] mainComponents = mainWindow.getComponents();
    final Component[] rootComponents = new Component[mainComponents.length + moreRoots.length];
    System.arraycopy(mainComponents, 0, rootComponents, 0, mainComponents.length);
    System.arraycopy(moreRoots, 0, rootComponents, mainComponents.length, moreRoots.length);
    final MouseWheelListener dpiAdjust = e -> {
      if ((e.getModifiersEx() & MouseWheelEvent.CTRL_DOWN_MASK) != 0) {
        final int clicks = e.getWheelRotation();
        if (clicks != 0) {
          adjustFontSizes(clicks, rootComponents);
          for (Component c : rootComponents) {
            updateUI(c);
          }
        }
      }
    };
    for (Component c : rootComponents) {
      c.addMouseWheelListener(dpiAdjust);
    }
  }

  /**
   * Setup default high DPI settings.
   * This uses the default graphics configuration of the default screen device for setup.
   * @return the font delta applied during setup
   */
  public static int setupDefaultHiDpi()
  {
    final GraphicsConfiguration config = GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().getDefaultConfiguration();
    return setupDefaultHiDpi(config);
  }

  /**
   * Setup default high DPI settings.
   * @param config graphics configuration used for setup
   * @return the font delta applied during setup
   */
  public static int setupDefaultHiDpi(@NotNull GraphicsConfiguration config)
  {
    final int fontSize = getMinUiFontSize();
    final Rectangle bounds = config.getBounds();
    final int scale = (int)Math.round(bounds.getHeight() / fontSize / DEFAULT_RELATIVE_FONT_SIZE);
    if (scale > 1) {
      final int delta = fontSize * (scale - 1);
      Debug.message("Initial setup: %0", delta);
      adjustFontSizes(delta);
      return delta;
    }
    return 0;
  }

  /**
   * Get a useful font size for the default screen's default graphics configuration
   * in order to display {@link #DEFAULT_NUM_ROWS} text rows
   * @return font size which would allow to display the given number of lines
   */
  public static int getUsefulFontSize()
  {
    return getUsefulFontSize(DEFAULT_NUM_ROWS);
  }

  /**
   * Get a useful font size for the given graphics configuration.
   * This uses the default configuration of the default screen for calculation.
   * @param nRows  number of text rows expected to be visible on screen
   * @return font size which would allow to display the given number of lines
   */
  public static int getUsefulFontSize(int nRows)
  {
    return getUsefulFontSize(GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().getDefaultConfiguration(),
                             nRows);
  }

  /**
   * Get a useful font size for the given graphics configuration.
   * This uses {@link #DEFAULT_NUM_ROWS} as number of visible rows.
   * @param config graphics configuration
   * @return font size which would allow to display the default number of text rows
   */
  public static int getUsefulFontSize(@NotNull GraphicsConfiguration config)
  {
    return getUsefulFontSize(config, DEFAULT_NUM_ROWS);
  }

  /**
   * Get a useful font size for the given graphics configuration.
   * @param config graphics configuration
   * @param nRows  number of text rows expected to be visible on screen
   * @return font size which would allow to display the given number of text rows
   */
  public static int getUsefulFontSize(@NotNull GraphicsConfiguration config, int nRows)
  {
    final Rectangle bounds = config.getBounds();
    return (int)Math.round(3 * bounds.getHeight() / (4 * nRows));
  }

  /**
   * Load a potentially scalable icon.
   * @param resourcePath internal resource path to image
   * @return icon
   */
  @Nullable
  public static Icon loadIconResource(@NotNull String resourcePath)
  {
    if (resourcePath.toLowerCase().endsWith(".vic")) {
      // special vector icon
      try {
        if (resourcePath.indexOf('|') < 0) {
          resourcePath = DEFAULT_ICON_LOAD_CLASS + "|" + resourcePath;
        }
        final URL url = Utility.openResourceViaClass(resourcePath);
        if (url == null) {
          Debug.error("Resource unavailable: %s", resourcePath);
          return null;
        }
        try (InputStream is = url.openStream()) {
          final VectorImage image = SimpleVectorImageReader.loadImage(is);
          return new DynamicallyScaledIcon(image, "ToolBar.font", 4./3.);
        }
      } catch (Exception e) {
        Debug.error(e);
        return null;
      }
    }
    final Image img = Utility.loadImage(resourcePath);
    if (img != null) {
      return new ImageIcon(img);
    }
    return null;
  }

  /**
   * Get the inner rectangle of a component which is available for drawing.
   * This takes care of possible insets in Swing components.
   * @param comp component for which the draw rectangle is requested
   * @return draw rectangle of component
   */
  @NotNull
  public static Rectangle getDrawableViewport(@NotNull Component comp)
  {
    return comp instanceof JComponent
            ? getDrawableViewport((JComponent)comp)
            : new Rectangle(comp.getSize());
  }

  /**
   * Get the inner rectangle of a Swing component which is available for drawing.
   * This takes care of possible insets.
   * @param comp component for which the draw rectangle is requested
   * @return draw rectangle of component
   */
  @NotNull
  public static Rectangle getDrawableViewport(@NotNull JComponent comp)
  {
    final Insets insets = comp.getInsets();
    final int w = comp.getWidth();
    final int h = comp.getHeight();
    return new Rectangle(insets.left, insets.top,
                         w - insets.left - insets.right,
                         h - insets.top - insets.bottom);
  }

  /**
   * Get a transformation which fits a source rectangle into a target rectangle with uniform
   * scaling.
   * <p>
   * This is most useful for calculating view transformations for fitting a bounding rectangle
   * into a screen rectangle.
   * @param source  source rectangle
   * @param target  target rectangle
   * @param margin  margin subtracted from all borders of target rectangle, use {@code 0.0} for no margin
   * @return fit transformation
   * @see #viewTransform(Rectangle2D, JComponent, double)
   */
  @NotNull
  public static AffineTransform fitTransform(@NotNull Rectangle2D source,
                                             @NotNull Rectangle2D target,
                                             double margin)
  {
    return fitTransform(source, target, margin, false);
  }

  /**
   * Get a transformation which fits a source rectangle into a target rectangle with uniform
   * scaling.
   * <p>
   * This is most useful for calculating view transformations for fitting a bounding rectangle
   * into a screen rectangle.
   * @param source  source rectangle
   * @param target  target rectangle
   * @param margin  margin subtracted from all borders of target rectangle, use {@code 0.0} for no margin
   * @param invertY invert the Y axis?
   * @return fit transformation
   * @see #viewTransform(Rectangle2D, JComponent, double)
   */
  @NotNull
  public static AffineTransform fitTransform(@NotNull Rectangle2D source,
                                             @NotNull Rectangle2D target,
                                             double margin,
                                             boolean invertY)
  {
    return fitTransform(source, target, margin, invertY, true);
  }

  /**
   * Get a transformation which fits a source rectangle into a target rectangle with uniform
   * scaling.
   * <p>
   * This is most useful for calculating view transformations for fitting a bounding rectangle
   * into a screen rectangle.
   * @param source  source rectangle
   * @param target  target rectangle
   * @param margin  margin subtracted from all borders of target rectangle, use {@code 0.0} for no margin
   * @param isotrop scale the same in both directions?
   * @param invertY invert the Y axis?
   * @return fit transformation
   * @see #viewTransform(Rectangle2D, JComponent, double)
   */
  @NotNull
  public static AffineTransform fitTransform(@NotNull Rectangle2D source,
                                             @NotNull Rectangle2D target,
                                             double margin,
                                             boolean invertY,
                                             boolean isotrop)
  {
    final double scaleX, scaleY;
    if (isotrop) {
      scaleX = scaleY = Math.min((target.getWidth() - 2 * margin) / source.getWidth(),
                                    (target.getHeight() - 2 * margin) / source.getHeight());
    }
    else {
      scaleX = (target.getWidth() - 2 * margin) / source.getWidth();
      scaleY = (target.getHeight() - 2 * margin) / source.getHeight();
    }
    final AffineTransform transform = new AffineTransform();
    transform.concatenate(AffineTransform.getTranslateInstance(target.getCenterX(),
                                                               target.getCenterY()));
    transform.concatenate(AffineTransform.getScaleInstance(scaleX,
                                                           invertY ? -scaleY : scaleY));
    transform.concatenate(AffineTransform.getTranslateInstance(-source.getCenterX(),
                                                               -source.getCenterY()));
    return transform;
  }

  /**
   * Get a transformation which fits a given source rectangle into a Swing component
   * using uniform scaling.
   * <p>
   * This especially takes care of possible borders applied to {@code target}.
   * @param source  source rectangle
   * @param target  target rectangle
   * @param margin  margin subtracted from all borders of target rectangle, use {@code 0.0} for no margin
   * @return fit transformation
   * @see #viewTransform(Rectangle2D, JComponent, double)
   * @see #getDrawableViewport(JComponent)
   */
  @NotNull
  public static AffineTransform viewTransform(@NotNull Rectangle2D source,
                                              @NotNull JComponent target,
                                              double margin)
  {
    return viewTransform(source, target, margin, false);
  }

  /**
   * Get a transformation which fits a given source rectangle into a Swing component
   * using uniform scaling.
   * <p>
   * This especially takes care of possible borders applied to {@code target}.
   * @param source  source rectangle
   * @param target  target rectangle
   * @param margin  margin subtracted from all borders of target rectangle, use {@code 0.0} for no margin
   * @param invertY invert the Y axis?
   * @return fit transformation
   * @see #viewTransform(Rectangle2D, JComponent, double)
   * @see #getDrawableViewport(JComponent)
   */
  @NotNull
  public static AffineTransform viewTransform(@NotNull Rectangle2D source,
                                              @NotNull JComponent target,
                                              double margin,
                                              boolean invertY)
  {
    return viewTransform(source, target, margin, invertY, true);
  }

  /**
   * Get a transformation which fits a given source rectangle into a Swing component
   * using uniform scaling.
   * <p>
   * This especially takes care of possible borders applied to {@code target}.
   * @param source  source rectangle
   * @param target  target rectangle
   * @param margin  margin subtracted from all borders of target rectangle, use {@code 0.0} for no margin
   * @param invertY invert the Y axis?
   * @param isotrop scale the same in both directions?
   * @return fit transformation
   * @see #viewTransform(Rectangle2D, JComponent, double)
   * @see #getDrawableViewport(JComponent)
   */
  @NotNull
  public static AffineTransform viewTransform(@NotNull Rectangle2D source,
                                              @NotNull JComponent target,
                                              double margin,
                                              boolean invertY,
                                              boolean isotrop)
  {
    return fitTransform(source, getDrawableViewport(target), margin, invertY, isotrop);
  }

  /**
   * Workaround for changed behavior of modifiers handling.
   * {@link MouseEvent#getModifiers()} contains the relevant
   * buttons even in the release case, while {@link MouseEvent#getModifiersEx()}
   * does not. This method includes the button in the modifiers
   * in its down state even although this is is basically wrong.
   * This allows old logic to still do its work.
   * @param event mouse event
   * @return result of {@link MouseEvent#getModifiersEx()} with
   *         the down flag for the button of interest
   */
  public static int getModifiersExWithButton(@NotNull MouseEvent event)
  {
    int mod = event.getModifiersEx();
    switch (event.getButton()) {
    case MouseEvent.BUTTON1:
      mod |= MouseEvent.BUTTON1_DOWN_MASK;
      break;
    case MouseEvent.BUTTON2:
      mod |= MouseEvent.BUTTON2_DOWN_MASK;
      break;
    case MouseEvent.BUTTON3:
      mod |= MouseEvent.BUTTON3_DOWN_MASK;
      break;
    }
    return mod;
  }

  /**
   * Invoke a GUI related task much later.
   * This will use a recursion count of {@code 8}.
   * @param task task to perform much later
   * @see #invokeMuchLater(int, Runnable)
   */
  public static void invokeMuchLater(@NotNull Runnable task)
  {
    invokeMuchLater(8, task);
  }

  /**
   * Invoke a GUI related task much later.
   * This will make the given number of reinvocations using
   * {@link SwingUtilities#invokeLater(Runnable)} until the
   * task is performed. Using {@code 1} for {@code recursions}
   * will make this method behave like
   * @param recursions number of reinvocations until {@code task} is performed
   * @param task       task to perform much later
   */
  public static void invokeMuchLater(int recursions,
                                     @NotNull Runnable task)
  {
    if (recursions <= 1) {
      SwingUtilities.invokeLater(task);
    }
    final Runnable run = new Runnable()
    {
      int counter = recursions;
      @Override
      public void run()
      {
        if (--counter > 0) {
          SwingUtilities.invokeLater(this);
        }
        else {
          SwingUtilities.invokeLater(task);
        }
      }
    };

    SwingUtilities.invokeLater(run);
  }

  /**
   * Get the part of the screen available for user application windows.
   * @param graphicsConfiguration graphics configuration of the screen of interest
   * @return screen viewport
   */
  @NotNull
  public static Rectangle getScreenViewport(@NotNull GraphicsConfiguration graphicsConfiguration)
  {
    final Rectangle screen = graphicsConfiguration.getBounds();
    final Insets screenInsets = Toolkit.getDefaultToolkit().getScreenInsets(graphicsConfiguration);
    screen.x += screenInsets.left;
    screen.y += screenInsets.top;
    screen.width -= screenInsets.left + screenInsets.right;
    screen.height -= screenInsets.top + screenInsets.bottom;
    return screen;
  }

  /**
   * A dictionary with rendering hints providing default settings for various properties.
   * It provides settings for most but not all properties.
   * Contained are rendering hints for
   * <ul>
   *   <li>{@link RenderingHints#KEY_RENDERING}</li>
   *   <li>{@link RenderingHints#KEY_ANTIALIASING}</li>
   *   <li>{@link RenderingHints#KEY_FRACTIONALMETRICS}</li>
   *   <li>{@link RenderingHints#KEY_TEXT_ANTIALIASING}</li>
   *   <li>{@link RenderingHints#KEY_INTERPOLATION}</li>
   *   <li>{@link RenderingHints#KEY_ALPHA_INTERPOLATION}</li>
   *   <li>{@link RenderingHints#KEY_COLOR_RENDERING}</li>
   * </ul>
   */
  public static final Dict<RenderingHints.Key, Object> DEFAULT_RENDER;
  /**
   * A dictionary with rendering hints providing nice settings for various properties.
   * It provides settings for most but not all properties.
   * Contained are rendering hints for
   * <ul>
   *   <li>{@link RenderingHints#KEY_RENDERING}</li>
   *   <li>{@link RenderingHints#KEY_ANTIALIASING}</li>
   *   <li>{@link RenderingHints#KEY_FRACTIONALMETRICS}</li>
   *   <li>{@link RenderingHints#KEY_TEXT_ANTIALIASING}</li>
   *   <li>{@link RenderingHints#KEY_INTERPOLATION}</li>
   *   <li>{@link RenderingHints#KEY_ALPHA_INTERPOLATION}</li>
   *   <li>{@link RenderingHints#KEY_COLOR_RENDERING}</li>
   * </ul>
   */
  public static final Dict<RenderingHints.Key, Object> NICE_RENDER;
  /**
   * A dictionary with rendering hints providing fast settings for various properties.
   * It provides settings for most but not all properties.
   * Contained are rendering hints for
   * <ul>
   *   <li>{@link RenderingHints#KEY_RENDERING}</li>
   *   <li>{@link RenderingHints#KEY_ANTIALIASING}</li>
   *   <li>{@link RenderingHints#KEY_FRACTIONALMETRICS}</li>
   *   <li>{@link RenderingHints#KEY_TEXT_ANTIALIASING}</li>
   *   <li>{@link RenderingHints#KEY_INTERPOLATION}</li>
   *   <li>{@link RenderingHints#KEY_ALPHA_INTERPOLATION}</li>
   *   <li>{@link RenderingHints#KEY_COLOR_RENDERING}</li>
   * </ul>
   */
  public static final Dict<RenderingHints.Key, Object> FAST_RENDER;

  static {
    final Map<RenderingHints.Key, Object> dflt = new HashMap<>();
    final Map<RenderingHints.Key, Object> nice = new HashMap<>();
    final Map<RenderingHints.Key, Object> fast = new HashMap<>();

    dflt.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_DEFAULT);
    nice.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
    fast.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);

    dflt.put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_DEFAULT);
    nice.put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
    fast.put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);

    dflt.put(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_DEFAULT);
    nice.put(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
    fast.put(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_OFF);

    dflt.put(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_DEFAULT);
    nice.put(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    fast.put(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);

    dflt.put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
    nice.put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
    fast.put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_NEAREST_NEIGHBOR);

    dflt.put(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_DEFAULT);
    nice.put(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
    fast.put(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);

    dflt.put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_DEFAULT);
    nice.put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
    fast.put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_SPEED);

    DEFAULT_RENDER = Dict.viewMap(dflt);
    NICE_RENDER    = Dict.viewMap(nice);
    FAST_RENDER    = Dict.viewMap(fast);
  }

  /**
   * Prepare the rendering hints of a graphics context.
   * This will set each rendering hint from {@code hints} for the given graphics.
   * @param g2     graphics context
   * @param hints  rendering hints to set
   */
  public static void prepareGraphics(@NotNull Graphics2D g2,
                                     @NotNull Dict<RenderingHints.Key, ?> hints)
  {
    hints.forEachEntry(g2::setRenderingHint);
  }

  /**
   * Prepare a graphics context to use default settings for rendering.
   * @param g2 graphics context
   * @see #prepareGraphics(Graphics2D, Dict)
   * @see #prepareNiceGraphics(Graphics2D)
   * @see #prepareFastGraphics(Graphics2D)
   */
  public static void prepareDefaultGraphics(@NotNull Graphics2D g2)
  {
    prepareGraphics(g2, DEFAULT_RENDER);
  }

  /**
   * Prepare a graphics context to use high-quality rendering.
   * This may slow down painting.
   * @param g2 graphics context
   * @see #prepareGraphics(Graphics2D, Dict)
   * @see #prepareDefaultGraphics(Graphics2D)
   * @see #prepareFastGraphics(Graphics2D)
   */
  public static void prepareNiceGraphics(@NotNull Graphics2D g2)
  {
    prepareGraphics(g2, NICE_RENDER);
  }

  /**
   * Prepare a graphics context to use the fastest settings for rendering.
   * The resulting output may look ugly.
   * @param g2 graphics context
   * @see #prepareGraphics(Graphics2D, Dict)
   * @see #prepareDefaultGraphics(Graphics2D)
   * @see #prepareNiceGraphics(Graphics2D)
   */
  public static void prepareFastGraphics(@NotNull Graphics2D g2)
  {
    prepareGraphics(g2, FAST_RENDER);
  }

  /**
   * Set up a panel to have two columns of automatic width.
   * <p>
   * This is useful for displaying name-value pairs in cases where a table is not sufficient.
   * @param components components ordered left, right, left, right, ...
   * @return panel with the given setup
   */
  @NotNull
  public static JPanel simpleTwoColumnPanel(@NotNull JComponent ... components)
  {
    if (components.length % 2 != 0) {
      throw new IllegalArgumentException("Need an even number of components!");
    }
    return simpleTwoColumnPanel(Indexable.viewByIndex(components.length / 2,
                                                      idx -> Pair.createPair(components[2 * idx],
                                                                             components[2 * idx + 1])));
  }

  /**
   * Set up a panel to have two columns of automatic width from an indexable array of components.
   * <p>
   * This is useful for displaying name-value pairs in cases where a table is not sufficient.
   * @param components components ordered left, right, left, right, ...
   * @return panel with the given setup
   */
  @NotNull
  public static JPanel simpleTwoColumnPanelFrom(@NotNull Indexable<? extends JComponent> components)
  {
    if (components.size() % 2 != 0) {
      throw new IllegalArgumentException("Need an even number of components!");
    }
    return simpleTwoColumnPanel(Indexable.viewByIndex(components.size() / 2,
                                                      idx -> Pair.createPair(components.get(2 * idx),
                                                                             components.get(2 * idx + 1))));
  }

  /**
   * Set up a panel to have two columns of automatic width.
   * <p>
   * This is useful for displaying name-value pairs in cases where a table is not sufficient.
   * @param rows pairs of components for the rows of the panel
   * @return panel with the given setup
   */
  @NotNull
  public static JPanel simpleTwoColumnPanel(@NotNull Iterable<Pair<JComponent>> rows)
  {
    final GridBagLayout gbl = new GridBagLayout();
    final JPanel panel = new JPanel(gbl);

    setupSimpleTwoColumnPanel(panel, rows);

    return panel;
  }

  /**
   * Set up a panel to have two columns of automatic width.
   * <p>
   * This is useful for displaying name-value pairs in cases where a table is not sufficient.
   *
   * @param panelWithGridBagLayout panel, required to have a {@link GridBagLayout}
   * @param rows pairs of components for the rows of the panel
   */
  public static void setupSimpleTwoColumnPanel(@NotNull JPanel panelWithGridBagLayout,
                                               @NotNull Iterable<Pair<JComponent>> rows)
  {
    final GridBagLayout gbl;
    try {
      gbl = (GridBagLayout)panelWithGridBagLayout.getLayout();
    } catch (ClassCastException x) {
      throw new IllegalArgumentException("Panel is expected to have a grid bag layout!",
                                         x);
    }

    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.ipadx = 4;
    constraints.ipady = 4;
    constraints.insets = new Insets(6, 4, 6, 4);
    constraints.gridy = 0;
    constraints.gridheight = 1;
    constraints.weighty = 0;

    for (Pair<JComponent> row : rows) {
      final Component first = Types.notNullOr(row.first, JLabel::new);
      constraints.gridx     = 0;
      constraints.gridwidth = 1;
      constraints.weightx   = 0;
      constraints.fill      = GridBagConstraints.HORIZONTAL;
      constraints.anchor    = GridBagConstraints.NORTHWEST;
      gbl.setConstraints(first, constraints);
      panelWithGridBagLayout.add(first);

      final Component second = Types.notNullOr(row.second, JLabel::new);
      ++constraints.gridx;
      constraints.anchor    = GridBagConstraints.NORTHEAST;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.weightx   = 1;
      constraints.fill      = GridBagConstraints.BOTH;
      gbl.setConstraints(second, constraints);
      panelWithGridBagLayout.add(second);

      ++constraints.gridy;
    }
  }

  /**
   * Create a panel to have a defined number of columns of automatic width.
   *
   * @param numColumns number of columns
   * @param components components to add, the number has to be a multiple of {@code numColums},
   *                   use {@code null} to keep a cell empty
   * @return panel with the given components organized in columns
   */
  @NotNull
  public static JPanel simpleMultiColumnPanel(int numColumns,
                                              @NotNull JComponent ... components)
  {
    final JPanel panel = new JPanel(new GridBagLayout());
    setupSimpleMultiColumnPanel(panel, numColumns, components);
    return panel;
  }

  /**
   * Set up a panel to have a defined number of columns of automatic width.
   *
   * @param panelWithGridBagLayout panel, required to have a {@link GridBagLayout}
   * @param numColumns number of columns
   * @param components components to add, the number has to be a multiple of {@code numColums},
   *                   use {@code null} to keep a cell empty
   */
  public static void setupSimpleMultiColumnPanel(@NotNull JPanel panelWithGridBagLayout,
                                                 int numColumns,
                                                 @NotNull JComponent ... components)
  {
    setupSimpleMultiColumnPanel(panelWithGridBagLayout, numColumns, Countable.viewArray(components));
  }

  /**
   * Create a panel to have a defined number of columns of automatic width.
   *
   * @param numColumns number of columns, at least {@code 1}
   * @param components components to add, the number has to be a multiple of {@code numColums},
   *                   use {@code null} to keep a cell empty
   * @return panel with the given components organized in columns
   */
  @NotNull
  public static JPanel createSimpleMultiColumnPanel(int numColumns,
                                                    @NotNull Countable<? extends JComponent> components)
  {
    final JPanel panel = new JPanel(new GridBagLayout());
    setupSimpleMultiColumnPanel(panel, numColumns, components);
    return panel;
  }

  /**
   * Set up a panel to have a defined number of columns of automatic width.
   *
   * @param panelWithGridBagLayout panel, required to have a {@link GridBagLayout}
   * @param numColumns number of columns, at least {@code 1}
   * @param components components to add, the number has to be a multiple of {@code numColums},
   *                   use {@code null} to keep a cell empty
   */
  public static void setupSimpleMultiColumnPanel(@NotNull JPanel panelWithGridBagLayout,
                                                 int numColumns,
                                                 @NotNull Countable<? extends JComponent> components)
  {
    final GridBagLayout gbl;
    try {
      gbl = (GridBagLayout)panelWithGridBagLayout.getLayout();
    } catch (ClassCastException x) {
      throw new IllegalArgumentException("Panel is expected to have a grid bag layout!",
                                         x);
    }
    if (numColumns <= 0) {
      throw new IllegalArgumentException("Need at least one column, but got "+numColumns);
    }
    if (components.size() % numColumns != 0) {
      throw new IllegalArgumentException(String.format("Number of components (%d) is not a multiple of number of columns (%d)!",
                                                       components.size(), numColumns));
    }

    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.ipadx = 4;
    constraints.ipady = 4;
    constraints.insets = new Insets(6, 4, 6, 4);
    constraints.gridy = -1;
    constraints.fill = GridBagConstraints.HORIZONTAL;

    int idx = 0;
    for (JComponent comp : components.view(cp -> Types.notNullOr(cp, JLabel::new))) {
      final int column = idx % numColumns;
      constraints.gridx = column;
      if (column == numColumns - 1) {
        constraints.gridwidth = GridBagConstraints.REMAINDER;
      }
      else {
        if (column == 0) {
          ++constraints.gridy;
        }
        constraints.gridwidth = 1;
      }
      gbl.setConstraints(comp, constraints);
      panelWithGridBagLayout.add(comp);
      ++idx;
    }
  }

  /**
   * Create a frame with a central component.
   * This is thought for quick-shot visual programs.
   * The frame is disposed on close.
   * @param title     frame title
   * @param component central component
   * @param width     frame width (use {@code 0} for packed)
   * @param height    frame height (use {@code 0} for packed
   * @return the created frame
   */
  @NotNull
  public static JFrame simpleApplicationFrame(@NotNull String title,
                                              @NotNull JComponent component,
                                              int width, int height)
  {
    final JFrame frame = new JFrame(title);
    frame.getContentPane().add(component);
    frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
    if (width <= 0  ||  height <= 0) {
      frame.pack();
    }
    else {
      frame.setSize(width, height);
    }
    return frame;
  }

  /**
   * Show a frame with a central component.
   * This is thought for quick-shot visual programs.
   * The frame is disposed on close.
   * @param title     frame title
   * @param component central component
   * @param width     frame width (use {@code 0} for packed)
   * @param height    frame height (use {@code 0} for packed
   * @return the created frame
   */
  @NotNull
  public static JFrame showSimpleApplicationFrame(@NotNull String title,
                                                  @NotNull JComponent component,
                                                  int width, int height)
  {
    final JFrame jFrame = simpleApplicationFrame(title, component, width, height);
    jFrame.setVisible(true);
    return jFrame;
  }

  /**
   * Build a row header view.
   * @param table   table for which the row header is created
   * @param headers  headers to set, should have the same number of rows as table
   * @return a JList which can be set as {@link JScrollPane#setRowHeaderView(Component) row header view component}
   */
  @NotNull
  public static JList<String> buildRowHeader(@NotNull final JTable table,
                                             @NotNull final Vector<String> headers)
  {
    final ListModel<String> listModel = new AbstractListModel<String>()
    {
      private static final long serialVersionUID = -772559910860527115L;

      public int getSize()
      {
        return headers.size();
      }

      public String getElementAt(int index)
      {
        return headers.get(index);
      }
    };

    final JList<String> rowHeader = new JList<>(listModel);
    rowHeader.setOpaque(false);

    final RowHeaderRenderer cellRenderer = new RowHeaderRenderer(table);
    int maxWidth = 0;
    int index = 0;
    for (String header : headers) {
      Component c = cellRenderer.getListCellRendererComponent(rowHeader,
                                                              header,
                                                              index++,
                                                              false,
                                                              false);
      maxWidth = Math.max(c.getPreferredSize().width, maxWidth);
    }
    rowHeader.setPreferredSize(new Dimension(maxWidth, rowHeader.getPreferredSize().height));
    rowHeader.setCellRenderer(cellRenderer);
    rowHeader.setBackground(table.getBackground());
    rowHeader.setForeground(table.getForeground());
    return rowHeader;
  }

  static class RowHeaderRenderer
          extends JLabel
          implements ListCellRenderer<String>
  {
    private static final long serialVersionUID = -8065593795296883073L;
    private final JTable table;

    RowHeaderRenderer(@NotNull JTable table)
    {
      this.table = table;
      JTableHeader header = this.table.getTableHeader();
      setOpaque(true);
      setBorder(BorderFactory.createCompoundBorder(UIManager.getBorder("TableHeader.cellBorder"),
                                                   BorderFactory.createEmptyBorder(4, 4, 4, 4)));
      setHorizontalAlignment(CENTER);
      setForeground(header.getForeground());
      setBackground(header.getBackground());
      setFont(header.getFont());
      setDoubleBuffered(true);
      setHorizontalAlignment(RIGHT);
    }

    @Override
    public Component getListCellRendererComponent(JList<? extends String> list, String value, int index,
                                                  boolean isSelected, boolean cellHasFocus)
    {
      setText(Types.notNull(value));
      setPreferredSize(null);
      setPreferredSize(new Dimension(getPreferredSize().width, table.getRowHeight(index)));
      list.repaint();
      return this;
    }

  }
}
