// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.gimmicks.swing.ResourcedAction;
import de.caff.i18n.swing.RJButton;
import de.caff.i18n.swing.RJPanel;
import de.caff.util.debug.Debug;
import de.caff.util.swing.SwingHelper;

import javax.swing.*;
import javax.swing.event.ListSelectionListener;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

/**
 *  An editor for URL lists.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class UrlListEditor
        extends AbstractBasicEditorProvider
{
  /** The property which is edited. */
  private final SwingUrlListPreferenceProperty property;
  /** The panel containing the editor widgets. */
  private final JPanel panel;
  /** A list of strings. */
  private final JList<String> list;

  @Nullable
  private final java.util.List<String> startList;
  /** URL selection mode. */
  private final SwingUrlListPreferenceProperty.UrlSelectionMode urlSelectionMode;

  private DefaultListModel<String> getModel()
  {
    return (DefaultListModel<String>)list.getModel();
  }

  /**
   * Action to add an entry to the list.
   */
  private final Action add    = new ResourcedAction("tbAdd") {
    private static final long serialVersionUID = 55118404268663770L;

    @Override
    public void actionPerformed(ActionEvent e)
    {
      UrlDialog dialog;
      Component root = SwingUtilities.getRoot(panel);
      if (root instanceof Dialog) {
        dialog = new UrlDialog((Dialog)root, null, urlSelectionMode);
      }
      else if (root instanceof Frame) {
        dialog = new UrlDialog((Frame)root, null, urlSelectionMode);
      }
      else {
        dialog = new UrlDialog(null, urlSelectionMode);
      }
      dialog.setVisible(true);
      String url = dialog.getUrl();
      if (url != null) {
        getModel().addElement(url);
      }
    }

    @Override
    protected Object clone() throws CloneNotSupportedException
    {
      return super.clone();
    }
  };
  private final Action delete = new ResourcedAction("tbDelete") {
    private static final long serialVersionUID = -6218395420075667584L;

    @Override
    public void actionPerformed(ActionEvent e)
    {
      getModel().removeElementAt(list.getSelectedIndex());
    }

    @Override
    protected Object clone() throws CloneNotSupportedException
    {
      return super.clone();
    }
  };
  private final Action change = new ResourcedAction("tbChange") {
    private static final long serialVersionUID = -4153188207958211070L;

    @Override
    public void actionPerformed(ActionEvent e)
    {
      UrlDialog dialog;
      Component root = SwingUtilities.getRoot(panel);
      if (root instanceof Dialog) {
        dialog = new UrlDialog((Dialog)root, list.getSelectedValue(), urlSelectionMode);
      }
      else if (root instanceof Frame) {
        dialog = new UrlDialog((Frame)root, list.getSelectedValue(), urlSelectionMode);
      }
      else {
        dialog = new UrlDialog(list.getSelectedValue(), urlSelectionMode);
      }
      dialog.setVisible(true);
      String url = dialog.getUrl();
      if (url != null) {
        getModel().setElementAt(url, list.getSelectedIndex());
      }
    }

    @Override
    protected Object clone() throws CloneNotSupportedException
    {
      return super.clone();
    }
  };
  private final Action up     = new ResourcedAction("tbUp") {
    private static final long serialVersionUID = 8893706601703876660L;

    @Override
    public void actionPerformed(ActionEvent e)
    {
      int index = list.getSelectedIndex();
      swap(index, index-1);
    }

    @Override
    protected Object clone() throws CloneNotSupportedException
    {
      return super.clone();
    }
  };
  private final Action down   = new ResourcedAction("tbDown") {
    private static final long serialVersionUID = -6479486919655281546L;

    @Override
    public void actionPerformed(ActionEvent e)
    {
      int index = list.getSelectedIndex();
      swap(index, index+11);
    }

    @Override
    protected Object clone() throws CloneNotSupportedException
    {
      return super.clone();
    }
  };

  /**
   * Constructor.
   * @param property edited property
   * @param l        locale
   * @param urlSelectionMode URL selection mode
   * @param extraActions additional actions to be shown in toolbar
   */
  public UrlListEditor(@NotNull  SwingUrlListPreferenceProperty property,
                       Locale l,
                       @NotNull SwingUrlListPreferenceProperty.UrlSelectionMode urlSelectionMode,
                       @NotNull Action ... extraActions)
  {
    super(property, l);
    this.property = property;
    startList = new ArrayList<>(property.getUrlList());
    final Indexable<String> defaults = property.getDefaults();

    final int sep = SwingHelper.getUsefulFontSize();
    panel = new JPanel(new BorderLayout());
    final JPanel buttons = new JPanel(new GridLayout(0, 1));
    buttons.add(new JButton(add));
    buttons.add(new JButton(delete));
    buttons.add(Box.createVerticalStrut(sep));
    buttons.add(new JButton(change));
    buttons.add(Box.createVerticalStrut(sep));
    buttons.add(new JButton(up));
    buttons.add(new JButton(down));
    if (defaults != null) {
      buttons.add(Box.createVerticalStrut(sep));
      final Action resetToDefaults = new ResourcedAction("tbResetToDefaults")
      {
        private static final long serialVersionUID = -3427781287206689984L;

        @Override
        public void actionPerformed(ActionEvent e)
        {
          final Indexable<String> defaults = property.getDefaults();
          assert defaults != null;

          final DefaultListModel<String> model = getModel();
          model.removeAllElements();
          defaults.forEach(model::addElement);
        }
      };
      buttons.add(new JButton(resetToDefaults));
    }
    if (extraActions.length > 0) {
      buttons.add(Box.createVerticalStrut(sep));
      for (Action a : extraActions) {
        buttons.add(new JButton(a));
      }
    }
    panel.add(buttons, BorderLayout.EAST);
    list = new JList<>(createListModel(property.getUrlList()));
    panel.add(new JScrollPane(list), BorderLayout.CENTER);

    list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    list.addListSelectionListener(e -> enableButtons(e.getFirstIndex()));

    enableButtons(list.getSelectedIndex());
    this.urlSelectionMode = urlSelectionMode;
  }

  @NotNull
  protected static DefaultListModel<String> createListModel(@NotNull Collection<String> c)
  {
    final DefaultListModel<String> model = new DefaultListModel<>();
    for (String aC : c) {
      model.addElement(aC);
    }
    return model;
  }

  /**
   * Add a selection listener to the URL list.
   * @param listener URL list selection listener
   */
  public void addListSelectionListener(@NotNull ListSelectionListener listener)
  {
    list.addListSelectionListener(listener);
  }

  /**
   * Remove a selection listener to the URL list.
   * @param listener URL list selection listener
   */
  public void removeListSelectionListener(@NotNull ListSelectionListener listener)
  {
    list.removeListSelectionListener(listener);
  }

  private void swap(int index1, int index2)
  {
    final DefaultListModel<String> model = getModel();
    final String obj1 = model.getElementAt(index1);
    final String obj2 = model.getElementAt(index2);
    model.setElementAt(obj1, index2);
    model.setElementAt(obj2, index1);
  }

  private void enableButtons(int index)
  {
    if (index < 0) {
      delete.setEnabled(false);
      change.setEnabled(false);
      up.setEnabled(false);
      down.setEnabled(false);
    }
    else {
      delete.setEnabled(true);
      change.setEnabled(true);
      up.setEnabled(index > 0);
      down.setEnabled(index < list.getModel().getSize()-1);
    }
  }

  /**
   * Get a component for editing.
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return panel;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    final DefaultListModel<String> model = getModel();
    model.removeAllElements();
    for (String s : property.getUrlList()) {
      model.addElement(s);
    }
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    final DefaultListModel<String> model = getModel();
    Object[] array = model.toArray();
    Collection<String> urlList = new ArrayList<>(array.length);
    for (Object a: array) {
      urlList.add(a.toString());
    }
    property.setUrlList(urlList);
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
  }

  /**
   *  Helper dialog for editing URLs.
   */
  private class UrlDialog
          extends JDialog
  {
    private static final long serialVersionUID = -915106342148489332L;
    /** The field displaying the URL. */
    private JTextField urlField;
    /** The okay button. */
    private JButton ok;
    /** The result. */
    private String result;
    /**
     * Constructor.
     * @param owner dialog owner
     * @param url   url to start with
     * @param urlSelectionMode URL selection mode
     */
    UrlDialog(@Nullable Dialog owner,
              @Nullable String url,
              @NotNull SwingUrlListPreferenceProperty.UrlSelectionMode urlSelectionMode)
    {
      super(owner);
      init(url, urlSelectionMode);
      setLocationRelativeTo(owner);
    }

    /**
     * Constructor.
     * @param owner dialog owner
     * @param url   url to start with
     * @param urlSelectionMode URL selection mode
     */
     UrlDialog(@Nullable Frame owner,
               @Nullable String url,
               @NotNull SwingUrlListPreferenceProperty.UrlSelectionMode urlSelectionMode)
    {
      super(owner);
      init(url, urlSelectionMode);
      setLocationRelativeTo(owner);
    }

    /**
     * Constructor.
     * @param url   url to start with
     * @param urlSelectionMode URL selection mode
     */
     UrlDialog(@Nullable String url,
               @NotNull SwingUrlListPreferenceProperty.UrlSelectionMode urlSelectionMode)
    {
      init(url, urlSelectionMode);
    }

    /**
     * Internally used to set up the dialog.
     * @param url  the URL to display
     * @param urlSelectionMode URL selection mode
     */
    private void init(String url, @NotNull SwingUrlListPreferenceProperty.UrlSelectionMode urlSelectionMode)
    {
      setModal(true);
      setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
      setTitle(property.getDialogTitle());
      urlField = new JTextField(url == null ? "" : url, 80);
      final JButton urlButton = property.getDialogButton();
      final JPanel buttonPanel = new RJPanel(new BorderLayout());
      buttonPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(8, 8, 8, 8),
                                                         BorderFactory.createEtchedBorder()));
      ok = new RJButton("Ok");
      final JButton cancel = new RJButton("Cancel");
      Box buttons = Box.createHorizontalBox();

      getContentPane().setLayout(new BorderLayout());
      buttonPanel.add(property.getDialogLabel(),
                      BorderLayout.WEST);
      buttonPanel.add(urlField, BorderLayout.CENTER);
      buttonPanel.add(urlButton, BorderLayout.EAST);
      buttons.add(Box.createHorizontalGlue());
      buttons.add(ok);
      buttons.add(cancel);
      buttons.add(Box.createHorizontalGlue());
      getContentPane().add(buttonPanel,   BorderLayout.NORTH);
      getContentPane().add(buttons, BorderLayout.SOUTH);

      urlButton.addActionListener(e -> {
        final File dir = extractFile();
        final JFileChooser chooser = new JFileChooser(dir);
        chooser.setMultiSelectionEnabled(false);
        chooser.setFileSelectionMode(urlSelectionMode.getFileSelectionMode());
        chooser.setFileFilter(property.getDialogFileFilter());
        if (chooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
          final File file = chooser.getSelectedFile();
          if (file != null) {
            try {
              urlField.setText(fileToUrl(file).toString());
            } catch (MalformedURLException x) {
              // how can this happen?
              Debug.error(x);
            }
          }
        }
      });

      urlField.addCaretListener(e -> checkUrl());

      ok.addActionListener(e -> {
        try {
          result = extractURL().toString();
        } catch (MalformedURLException x) {
          Debug.error("Okay button should not be active!");
        }
        dispose();
      });

      cancel.addActionListener(e -> dispose());

      checkUrl();
      pack();
    }

    @NotNull
    private URL extractURL()
            throws MalformedURLException
    {
      final String txt = urlField.getText().trim();
      try {
        return new URL(txt);
      } catch (MalformedURLException x) {
        // try again assuming the entry is a file
        final File file = new File(txt);
        if (file.exists()  &&  urlSelectionMode.isAllowed(file)) {
          return file.toURI().toURL();
        }
        throw x;
      }
    }

    @NotNull
    private File extractFile()
    {
      String path = urlField.getText().trim();
      if (path.startsWith("file:")) {
        try {
          final URL url = extractURL();
          path = url.getPath();
        } catch (MalformedURLException e) {
          Debug.error(e);
          // fallthrough
        }
      }
      return new File(path);
    }

    private void checkUrl()
    {
      try {
        extractURL();
        ok.setEnabled(true);
        urlField.setBackground(Color.white);
      } catch (MalformedURLException x) {
        ok.setEnabled(false);
        urlField.setBackground(new Color(0xff, 0xc0, 0xc0));
      }
    }

    /**
     * Get the URL which was set.
     * @return URL or {@code null} if the user canceled the dialog
     */
    public String getUrl()
    {
      return result;
    }
  }

  @NotNull
  private static URL fileToUrl(@NotNull File file)
          throws MalformedURLException
  {
    // broken on Unix, missing a slash after "file:"
    return file.toURI().toURL();
  }
}
