// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.gimmicks.swing.PropagateEnableScrollPane;

import javax.swing.*;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import java.awt.*;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Locale;

/**
 *  Editor provider for multi-line string properties.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class TextEditor
        extends AbstractBasicEditorProvider
        implements PropertyChangeListener
{
  private final SwingTextProperty property;
  private final JTextArea textArea;

  /**
   *  Constructor.
   *  @param property  string property for which this editor provider is created
   *  @param autoSave  automatically save changes
   *  @param l         locale
   */
  public TextEditor(@NotNull SwingTextProperty property, boolean autoSave, Locale l)
  {
    super(property, l);
    this.property = property;
    textArea = new JTextArea(property.getValue(),
                             property.getRows(),
                             property.getColumns());

    property.addValueChangeListenerWeakly(this);
    if (autoSave) {
      textArea.getDocument().addDocumentListener(new DocumentListener()
      {
        @Override
        public void insertUpdate(DocumentEvent e)
        {
          saveLater();
        }

        @Override
        public void removeUpdate(DocumentEvent e)
        {
          saveLater();
        }

        @Override
        public void changedUpdate(DocumentEvent e)
        {
          saveLater();
        }
      });
    }
    textArea.addFocusListener(new FocusAdapter()
    {
      @Override
      public void focusGained(FocusEvent e)
      {
        textArea.selectAll();
      }
    });
  }
  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    JScrollPane scrollPane = new PropagateEnableScrollPane(textArea);
    Dimension size = textArea.getPreferredSize();
    scrollPane.setMinimumSize(new Dimension(size.width/2, size.height));
    return scrollPane;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    textArea.setText(property.getValue());
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    property.setValue(textArea.getText());
  }

  /**
   * Save later.
   */
  public void saveLater()
  {
    SwingUtilities.invokeLater(this::save);
  }

  /**
   * This method gets called when a bound property is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the property that has changed.
   */
  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    if (!textArea.getText().equals(evt.getNewValue().toString())) {
      textArea.setText(evt.getNewValue().toString());
    }
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
    property.removeValueChangeListener(this);
  }
}
