// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.util.Iterator;
import java.util.Locale;

/**
 *  A editor provider for a preference group which displays each peference property in a tab of its own.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class TabbedGroupEditor
        extends AbstractBasicGroupEditor
{
  /** The tabbed pane which holds the properties. */
  private final JTabbedPane tabbedPane;

  /**
   *  Constructor.
   *  @param property group property
   *  @param l        locale
   */
  public TabbedGroupEditor(@NotNull final EditablePropertyGroup<?> property, @Nullable Locale l)
  {
    super(property.getNameTag(), l);

    tabbedPane = new JTabbedPane(SwingConstants.TOP, JTabbedPane.SCROLL_TAB_LAYOUT);
    for (Iterator<? extends EditableProperty> iterator = property.propertyIterator();  iterator.hasNext();  ) {
      EditableProperty epp = iterator.next();
      EditorProvider provider = epp.getEditorProvider(l);
      addEditorProvider(provider);

      tabbedPane.addTab(provider.getTitleText(), new JScrollPane(provider.getEditor()));
      tabbedPane.setToolTipTextAt(tabbedPane.getTabCount()-1, provider.getToolTipText());
    }
    tabbedPane.addChangeListener(e -> property.setTabIndex(tabbedPane.getSelectedIndex()));
    tabbedPane.setSelectedIndex(property.getTabIndex());
  }

  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return tabbedPane;
  }
}
