// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.i18n.I18n;
import de.caff.i18n.swing.RJButton;
import de.caff.i18n.swing.RJLabel;
import de.caff.util.Utility;
import de.caff.util.settings.UrlListPreferenceProperty;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.io.File;
import java.util.*;
import java.util.prefs.Preferences;

/**
 *  Editable URL list preference property.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingUrlListPreferenceProperty
        extends AbstractBasicSimpleEditableChangeableItem
        implements UrlListPreferenceProperty,
                   EditablePreferenceProperty
{
  /** I18n tag extension for title of associated dialog. */
  public static final String DIALOG_TITLE = "#diTitle";
  /** I18n tag extension for button of associated dialog. */
  public static final String DIALOG_BUTTON = "#diButton";
  /** I18n tag extension for label of associated dialog. */
  public static final String DIALOG_LABEL = "#diLabel";
  /** 118n tag extension for file filter definition. */
  public static final String DIALOG_FILTER = "#diFilter";

  /** Modes for local file URL selection. */
  public enum UrlSelectionMode
  {
    /** Allow both files and directories to be selected. */
    FilesAndDirectories(true, true, JFileChooser.FILES_AND_DIRECTORIES),
    /** Allow only files to be selected. */
    FilesOnly(true, false, JFileChooser.FILES_ONLY),
    /** Allow only directories to be selected. */
    DirectoriesOnly(false, true, JFileChooser.DIRECTORIES_ONLY);

    private final boolean allowsFiles;
    private final boolean allowsDirectories;
    private final int fileSelectionMode;

    /**
     * Constructor.
     * @param fileSelectionMode equivalent file selection mode of JFileChooser
     */
    UrlSelectionMode(boolean allowsFiles, boolean allowsDirectories,
                     int fileSelectionMode)
    {
      this.allowsFiles = allowsFiles;
      this.allowsDirectories = allowsDirectories;
      this.fileSelectionMode = fileSelectionMode;
    }

    /**
     * Get the file selection mode of JFileChooser which is equivalent to this URL selection mode.
     * @return file selection mode
     */
    public int getFileSelectionMode()
    {
      return fileSelectionMode;
    }

    /**
     * Check whether the given file object is allowed.
     * @param file either a file or a directory
     * @return {@code true}: this mode allows the object<br>
     *         {@code false}: this mode does not allow the object
     */
    public boolean isAllowed(@NotNull File file)
    {
      return file.isDirectory()
              ? allowsDirectories
              : allowsFiles;
    }
  }

  /** The preference key suffix for the number of entries. */
  private static final String PREF_KEY_SUFFIX_NR = "NR";
  private static final long serialVersionUID = 8150012278245474706L;

  @Nullable
  private final Indexable<String> defaults;

  /** The list of URLs. Elements are Strings. */
  @NotNull
  private List<String> urlList;

  /** The URL selection mode. */
  @NotNull
  private final UrlSelectionMode selectionMode;

  /**
   * Create a URL list with the given basic name and basic tag.
   * @param basicName basic name
   * @param baseTag   basic I18n tag
   * @param startList start list, if {@code null} there will be no "Defaults" button in the editor
   * @param selectionMode URL selection mode
   */
  public SwingUrlListPreferenceProperty(@NotNull String basicName,
                                        @NotNull String baseTag,
                                        @Nullable Collection<String> startList,
                                        @NotNull UrlSelectionMode selectionMode)
  {
    super(basicName, baseTag);
    this.selectionMode = selectionMode;

    urlList = startList == null
            ? new ArrayList<>()
            : new ArrayList<>(startList);

    this.defaults = startList == null
            ? null
            : Indexable.fromCollection(startList);
  }

  /**
   * Get the selection mode.
   * @return selection mode
   */
  @NotNull
  public UrlSelectionMode getSelectionMode()
  {
    return selectionMode;
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor component
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new UrlListEditor(this, l, selectionMode);
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    final String tagPrefix = getBasicName();
    int size = preferences.getInt(tagPrefix+PREF_KEY_SUFFIX_NR, -1);
    if (size == -1) {
      // previously unset, keep current
    }
    else {
      List<String> oldList = new ArrayList<>(urlList);
      urlList = new ArrayList<>(size);
      for (int index = 0;  index < size;  ++index) {
        String url = preferences.get(tagPrefix+index, null);
        if (url != null) {
          urlList.add(url);
        }
      }
      boolean changed;
      if (oldList.size() != urlList.size()) {
        changed = true;
      }
      else {
        Iterator<?> itOld = oldList.iterator();
        Iterator<?> itNew = urlList.iterator();
        changed = false;
        while (itOld.hasNext()  &&  itNew.hasNext()) {
          if (!itOld.next().equals(itNew.next())) {
            changed = true;
            break;
          }
        }
        changed = changed  ||  (itOld.hasNext() ^ itNew.hasNext());
      }
      if (changed) {
        fireValueChange(getBasicName(), oldList, urlList);
      }
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    final String tagPrefix = getBasicName();
    if (urlList.isEmpty()) {
      preferences.putInt(tagPrefix+PREF_KEY_SUFFIX_NR, 0);
    }
    else {
      preferences.putInt(tagPrefix+PREF_KEY_SUFFIX_NR, urlList.size());
      int index = 0;
      for (String url: urlList) {
        if (url != null) {
          preferences.put(tagPrefix+index, url);
        }
        ++index;
      }
    }
  }

  /**
   *  Add a URL to the end of the list.
   *  @param url string representing a URL
   */
  public void addUrl(String url)
  {
    addUrl(url, -1);
  }

  /**
   *  Insert a URL at the given position. If the position is invalid, the
   *  URL is added to the end.
   *  @param url string representing an URL
   *  @param pos insertion position
   */
  public void addUrl(String url, int pos)
  {
    List<String> oldList = new ArrayList<>(urlList);
    if (pos < 0  &&  pos >= urlList.size()) {
      urlList.add(url);
    }
    else {
      urlList.add(pos, url);
    }
    fireValueChange(getBasicName(), oldList, urlList);
  }

  /**
   *  Remove the URL at the given position.
   *  If the position is invalid, nothing happens.
   *  @param pos position to remove from the list
   */
  public void remove(int pos)
  {
    if (pos >= 0  &&  pos < urlList.size()) {
      List<String> oldList = new ArrayList<>(urlList);
      urlList.remove(pos);
      fireValueChange(getBasicName(), oldList, urlList);
    }
  }

  /**
   *  Exchange two entries in the URL list.
   *  @param pos1 position of first entry
   *  @param pos2 position of second entry
   */
  public void swap(int pos1, int pos2)
  {
    if (pos1 >= 0  &&  pos1 < urlList.size()  &&
        pos2 >= 0  &&  pos2 < urlList.size()  &&
        pos1 != pos2) {
      List<String> oldList = new ArrayList<>(urlList);
      Collections.swap(urlList, pos1, pos2);
      fireValueChange(getBasicName(), oldList, urlList);
    }
  }

  /**
   *  Get the collection of URLs.
   *  This also includes any URLs that are prepended or appended.
   *  @return URL list
   */
  @Override
  public Collection<String> getUrlList()
  {
    return Collections.unmodifiableCollection(urlList);
  }

  /**
   *  Set the URL list.
   *  @param newList new URL list
   */
  void setUrlList(@NotNull Collection<String> newList)
  {
    boolean changed;
    if (newList.size() != urlList.size()) {
      changed = true;
    }
    else {
      Iterator<?> itOld = urlList.iterator();
      Iterator<?> itNew = newList.iterator();
      changed = false;
      while (itOld.hasNext()  &&  itNew.hasNext()) {
        if (!itOld.next().equals(itNew.next())) {
          changed = true;
          break;
        }
      }
      changed = changed  ||  (itOld.hasNext() ^ itNew.hasNext());
    }
    Collection<String> oldList = urlList;
    urlList = new ArrayList<>(newList);
    if (changed) {
      fireValueChange(getBasicName(), oldList, urlList);
    }
  }

  /**
   * Get the defaults.
   * This is the initial list this preference property had by default.
   * @return default list, possibly empty, or {@code null} to have no reset button in the editor
   */
  @Nullable
  public Indexable<String> getDefaults()
  {
    return defaults;
  }

  /**
   * Get the title of the associated dialog.
   * @return dialog title
   */
  @NotNull
  String getDialogTitle()
  {
    try {
      return I18n.getString(getBaseTag() + DIALOG_TITLE);
    } catch (MissingResourceException x) {
      return I18n.getString("tiFontUrl");
    }
  }

  /**
   * Get the button of the associated dialog.
   * @return dialog button
   */
  @NotNull
  JButton getDialogButton()
  {
    try {
      return new RJButton(getBaseTag() + DIALOG_BUTTON);
    } catch (MissingResourceException x) {
      return new RJButton("btFontUrl");
    }
  }

  /**
   * Get the label text of the associated dialog.
   * @return dialog label
   */
  @NotNull
  JLabel getDialogLabel()
  {
    try {
      return new RJLabel(getBaseTag() + DIALOG_LABEL);
    } catch (MissingResourceException x) {
      return new RJLabel("lbFontUrl");
    }
  }

  /**
   * Get the file filter for the associated dialog.
   * @return file filter
   */
  @Nullable
  FileFilter getDialogFileFilter()
  {
    try {
      final String mask = I18n.getString(getBaseTag() + DIALOG_FILTER);
      if (!mask.isEmpty()) {
        final String[] parts = mask.split("\\|");
        return new FileFilter()
        {
          @Override
          public boolean accept(File f)
          {
            if (!selectionMode.isAllowed(f)) {
              return false;
            }
            if (f.isDirectory()) {
              // don't filter directory names
              return true;
            }
            for (String part : parts) {
              if (Utility.globEquals(part.toLowerCase(),
                                     f.getName().toLowerCase())) {
                return true;
              }
            }
            return false;
          }

          @Override
          public String getDescription()
          {
            return mask;
          }
        };
      }
    } catch (MissingResourceException x) {
    }
    return null;
  }
}
