// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.settings.Property;

import java.util.Locale;

/**
 * Simple one-line String property.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingTextProperty
        extends AbstractBasicSimpleEditableChangeableItem
        implements Property<String>
{
  private static final long serialVersionUID = -1880148426126237161L;
  /** Save values automatically? */
  protected final boolean autoSave;
  /** The string value. */
  protected String value;
  /** Rows of text area. */
  private final int rows;
  /** Columns of text area. */
  private final int columns;

  /**
   * Constructor.
   * @param basicName  basic name
   * @param baseTag    base tag
   * @param autoSave   automatically save changes?
   * @param value      text value
   * @param rows       rows of editor
   * @param columns    columns of editor
   */
  public SwingTextProperty(@NotNull String basicName,
                           @NotNull String baseTag,
                           boolean autoSave,
                           String value,
                           int rows, int columns)
  {
    super(basicName, baseTag);
    this.autoSave = autoSave;
    this.value = value;
    this.rows = rows;
    this.columns = columns;
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor component
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new TextEditor(this, autoSave, l);
  }

  /**
   *  Get the string value.
   *  @return string value
   */
  @Override
  public String getValue()
  {
    return value;
  }

  /**
   *  Set the value.
   *  @param value  string value
   */
  @Override
  public void setValue(String value)
  {
    if (value != null ? !value.equals(this.value) : this.value != null) {
      String oldValue = this.value;
      this.value = value;
      fireValueChange(getBasicName(), oldValue, value);
    }
  }

  /**
   * Get the rows which the editor should provide.
   * @return editor rows
   */
  public int getRows()
  {
    return rows;
  }

  /**
   * Get the colunns which the editor should provide.
   * @return editor columns
   */
  public int getColumns()
  {
    return columns;
  }
}
