// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.util.settings.StringPreferenceProperty;

import java.util.prefs.Preferences;

/**
 *  A preference property with a string value.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingStringPreferenceProperty
        extends SwingStringProperty
        implements StringPreferenceProperty,
                   EditablePreferenceProperty
{

  private static final long serialVersionUID = 7398495679907656012L;

  /**
   *  Constructor.
   *  @param basicName  basic name
   *  @param baseTag    basic i18n tag
   */
  public SwingStringPreferenceProperty(@NotNull String basicName,
                                       @NotNull String baseTag)
  {
    super(basicName, baseTag);
  }

  /**
   *  Constructor.
   *  @param basicName  basic name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   */
  public SwingStringPreferenceProperty(@NotNull String basicName,
                                       @NotNull String baseTag,
                                       @NotNull String value)
  {
    super(basicName, baseTag);
    this.value = value;
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    setValue(preferences.get(getBasicName(), value));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    if (value == null) {
      preferences.remove(getBasicName());
    }
    else {
      preferences.put(getBasicName(), value);
    }
  }

}
