// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.swing.RJMenu;
import de.caff.util.settings.AbstractBasicLocalizableChangeableItem;
import de.caff.util.settings.StringListPreferenceProperty;

import javax.swing.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.*;
import java.util.prefs.Preferences;

/**
 *  Hold a list of recently opened files.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingRecentFilesPreferenceProperty
  extends AbstractBasicLocalizableChangeableItem
  implements StringListPreferenceProperty
{
  private static final long serialVersionUID = 8713487176651453571L;

  /**
   *  Interface for loaders of recent files.
   */
  public static interface RecentFileLoader
  {
    /**
     *  Load a file selected from the recent file menu.
     *  @param path file path
     */
    public void loadRecentFile(String path);
  }

  /** The preference key suffix for the number of entries. */
  private static final String PREF_KEY_SUFFIX_NR = "NR";
  /** The maximum number of allowd entries. */
  public static final int MAXIMUM_STORE_SIZE = 10;

  /** The maximum number of entries stored. */
  private final int maxSize;
  /** The list of files. */
  private List<String> files = new LinkedList<>();

  /**
   *  Constructor.
   *  Uses {@link #MAXIMUM_STORE_SIZE} as allowed size for list.
   *  @param basicName  basic name (prefix)
   *  @param baseTag    basic i18n tag
   */
  public SwingRecentFilesPreferenceProperty(@NotNull String basicName, @NotNull String baseTag)
  {
    this(basicName, baseTag, MAXIMUM_STORE_SIZE);
  }

  /**
   *  Constructor.
   *  @param basicName  basic name (prefix)
   *  @param baseTag    basic i18n tag
   *  @param maxSize    maximum number of recent files stored, must be not more than {@link #MAXIMUM_STORE_SIZE}
   */
  public SwingRecentFilesPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, int maxSize)
  {
    super(basicName, baseTag);
    if (maxSize > MAXIMUM_STORE_SIZE) {
      throw new IllegalArgumentException("maxSize parameter is too large");
    }
    this.maxSize = maxSize;
  }

  /**
   *  Add a file path which becomes the most recent entry.
   *  @param filepath  filepath to add
   */
  public void addEntry(@NotNull String filepath)
  {
    Collection<String> oldFiles = new ArrayList<>(files);
    for (ListIterator<String> lit = files.listIterator();  lit.hasNext();  ) {
      if (filepath.equals(lit.next())) {
        lit.remove();
        break;
      }
    }
    if (files.size() == maxSize) {
      files.remove(maxSize-1);
    }
    files.add(0, filepath);
    fireValueChange(getBasicName(),
                    Collections.unmodifiableCollection(oldFiles),
                    Collections.unmodifiableCollection(files));
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    final String tagPrefix = getBasicName();
    int size = preferences.getInt(tagPrefix+PREF_KEY_SUFFIX_NR, -1);
    if (size == -1) {
      // previously unset, keep swing
    }
    else {
      if (size > maxSize) {
        size = maxSize;
      }
      List<String> oldList = new ArrayList<>(files);
      files = new ArrayList<>(size);
      for (int index = 0;  index < size;  ++index) {
        String path = preferences.get(tagPrefix+index, null);
        if (path != null) {
          files.add(path);
        }
      }
      boolean changed;
      if (oldList.size() != files.size()) {
        changed = true;
      }
      else {
        Iterator<?> itOld = oldList.iterator();
        Iterator<?> itNew = files.iterator();
        changed = false;
        while (itOld.hasNext()  &&  itNew.hasNext()) {
          if (!itOld.next().equals(itNew.next())) {
            changed = true;
            break;
          }
        }
        changed = changed  ||  (itOld.hasNext() ^ itNew.hasNext());
      }
      if (changed) {
        fireValueChange(getBasicName(),
                        Collections.unmodifiableCollection(oldList),
                        Collections.unmodifiableCollection(files));
      }
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    final String tagPrefix = getBasicName();
    preferences.putInt(tagPrefix+PREF_KEY_SUFFIX_NR, files.size());
    int index = 0;
    for (Iterator<String> iterator = files.iterator(); iterator.hasNext();  ++index) {
      String url = iterator.next();
      preferences.put(tagPrefix+index, url);
    }
  }

  /**
   * Get the collection of strings.
   *
   * @return String list
   */
  @NotNull
  @Override
  public List<String> getStringList()
  {
    return Collections.unmodifiableList(files);
  }

  /**
   *  Return the entry which was added most recently.
   *  @return latest entry or {@code null} if no entry was saved
   */
  @Nullable
  public String getLatestEntry()
  {
    return files.isEmpty() ? null : files.get(0);
  }

  /**
   *  Get a menu item representing the list of recent files.
   *  @param loader loader which loads the files if one is selected
   *  @param l locale
   *  @return menu item
   */
  @NotNull
  public JMenuItem getMenuItem(RecentFileLoader loader, Locale l)
  {
    SpecialMenu item = new SpecialMenu(loader, getBaseTag());
    item.setLocale(l);
    item.propertyChange(new PropertyChangeEvent(this, getBasicName(), null, files));
    addValueChangeListenerWeakly(item);
    return item;
  }

  /**
   *  A special menu which shows te recent files as popup.
   */
  private static class SpecialMenu
    extends RJMenu
    implements PropertyChangeListener
  {
    /**
     *  Special menu item calling the loader when activated.
     */
    private class FileMenuItem
      extends JMenuItem
      implements ActionListener
    {
      private static final long serialVersionUID = -1836115225804583659L;
      private final String filename;

      public FileMenuItem(String number, String filename)
      {
        super(number+" "+filename);
        this.filename = filename;
        addActionListener(this);
        setMnemonic(number.charAt(0));
      }

      /**
       * Invoked when an action occurs.
       */
      @Override
      public void actionPerformed(ActionEvent e)
      {
        loader.loadRecentFile(filename);
      }
    }

    private static final long serialVersionUID = 2253619552080450537L;

    /** Recent file loader to be called when file is selected. */
    private final RecentFileLoader loader;

    /**
     * Create a special menu.
     * @param loader   access to recent files
     * @param baseTag  basic I18n tag
     */
    public SpecialMenu(RecentFileLoader loader, String baseTag)
    {
      super(baseTag);
      this.loader = loader;
    }

    /**
     * This method gets called when a bound property is changed.
     *
     * @param evt A PropertyChangeEvent object describing the event source
     *            and the property that has changed.
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt)
    {
      removeAll();
      Collection<?> newFiles = (Collection)evt.getNewValue();
      int nr = 1;
      for (Iterator<?> iterator = newFiles.iterator(); iterator.hasNext(); ++nr) {
        String file = (String)iterator.next();
        final String number = Integer.toString(nr%10);
        add(new FileMenuItem(number, file));
      }
      setEnabled(!newFiles.isEmpty());
    }

  }
}
