// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.Locale;

/**
 *  Basic editable property for numbers.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public abstract class SwingNumberProperty
        extends AbstractBasicSimpleEditableChangeableItem
{
  private static final long serialVersionUID = 4359703239037082589L;
  /** Save changes immediately? */
  private final boolean autoSave;

  /**
   *  Create a integer preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param autoSave   save changes immediately?
   */
  public SwingNumberProperty(@NotNull String basicName,
                             @NotNull String baseTag,
                             boolean autoSave)
  {
    super(basicName, baseTag);
    this.autoSave = autoSave;
  }

  /**
   *  Get the minimal allowed value.
   *  @return minimal value
   */
  @NotNull
  public abstract Number getMinValueNumber();

  /**
   *  Get the maximal allowed value.
   *  @return maximal value
   */
  @NotNull
  public abstract Number getMaxValueNumber();

  /**
   *  Get the step used in steppers.
   *  @return step
   */
  @NotNull
  public abstract Number getStepNumber();

  /**
   *  Get the properties value.
   *  @return property value
   */
  @NotNull
  public abstract Number getValueNumber();

  /**
   *  Set the property value.
   *  @param value new value
   */
  public abstract void setValueNumber(@NotNull Number value);

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor components
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new NumberEditor(this, l, autoSave);
  }
}
