// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;
import de.caff.generics.function.Function1;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.i18n.swing.RJMenu;
import de.caff.util.debug.Debug;
import de.caff.util.settings.AbstractBasicLocalizableChangeableItem;
import de.caff.util.settings.StringListPreferenceProperty;

import javax.swing.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.*;
import java.util.prefs.Preferences;

/**
 * Allow to select the Look And Feel.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class SwingLookAndFeelPreferenceProperty
        extends AbstractBasicLocalizableChangeableItem
        implements StringListPreferenceProperty
{
  private static final long serialVersionUID = 6254879720735340597L;

  /** Property key for the look and feel name. */
  public static final String PROPERTY_LAF_NAME = "LAF_NAME";

  private static SwingLookAndFeelPreferenceProperty SINGLETON = null;

  private static final String KEY_LAF = "LookAndFeel";

  private static final List<UIManager.LookAndFeelInfo> LOOK_AND_FEELS = Arrays.asList(UIManager.getInstalledLookAndFeels());
  private int current;

  /**
   * Get the only instance of the laf property.
   * @return the instance
   */
  @NotNull
  public static SwingLookAndFeelPreferenceProperty getInstance()
  {
    synchronized (LOOK_AND_FEELS) {
      if (SINGLETON == null) {
        SINGLETON = new SwingLookAndFeelPreferenceProperty();
      }
      return SINGLETON;
    }
  }

  /**
   *  Constructor.
   */
  private SwingLookAndFeelPreferenceProperty()
  {
    super("PLAF", "ppLookAndFeel");
    setLookAndFeel(UIManager.getLookAndFeel().getName());
  }

  /**
   * Get the collection of strings.
   *
   * @return String list
   */
  @Override
  public Collection<String> getStringList()
  {
    return Types.map(LOOK_AND_FEELS, (Function1<String, UIManager.LookAndFeelInfo>)UIManager.LookAndFeelInfo::getName);
  }

  /**
   * Get the currently active LAF info.
   * @return LAF info
   */
  public UIManager.LookAndFeelInfo getCurrentLookAndFeelInfo()
  {
    return LOOK_AND_FEELS.get(current);
  }

  /**
   * Set the look and feel.
   * @param name name of look and feel
   */
  public void setLookAndFeel(@NotNull String name)
  {
    int i = 0;
    for (UIManager.LookAndFeelInfo laf : LOOK_AND_FEELS) {
      if (name.equalsIgnoreCase(laf.getName())) {
        setLookAndFeel(i);
        break;
      }
      ++i;
    }
  }

  private void setLookAndFeelClass(@NotNull String className)
  {
    int i = 0;
    for (UIManager.LookAndFeelInfo laf : LOOK_AND_FEELS) {
      if (className.equalsIgnoreCase(laf.getClassName())) {
        setLookAndFeel(i);
        break;
      }
      ++i;
    }
  }

  private void setLookAndFeel(int newCurrent)
  {
    if (newCurrent != current) {
      try {
        UIManager.setLookAndFeel(LOOK_AND_FEELS.get(newCurrent).getClassName());
        int oldValue = current;
        current = newCurrent;
        fireValueChange(PROPERTY_LAF_NAME,
                        LOOK_AND_FEELS.get(oldValue).getName(),
                        LOOK_AND_FEELS.get(current).getName());
      } catch (Exception e) {
        Debug.error(e);
      }
    }
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    String lafClass = preferences.get(KEY_LAF, null);
    if (lafClass == null) {
      return;
    }
    setLookAndFeelClass(lafClass);
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    preferences.put(KEY_LAF, LOOK_AND_FEELS.get(current).getClassName());
  }

  /**
   *  Create a menu item for this boolean property.
   *  @param l locale
   *  @return menu item
   */
  public JMenuItem createMenuItem(Locale l)
  {
    RJMenu menu = new RJMenu(getBaseTag());
    ButtonGroup group = new ButtonGroup();
    for (UIManager.LookAndFeelInfo info : LOOK_AND_FEELS) {
      SpecialMenuItem menuItem = new SpecialMenuItem(info, l);
      menu.add(menuItem);
      group.add(menuItem);
    }
    return menu;
  }

  /**
   * Helper class for creating a menu,
   */
  private class SpecialMenuItem
          extends JRadioButtonMenuItem
          implements PropertyChangeListener,
                     Localizable
  {

    private static final long serialVersionUID = -4293561993162568951L;
    private final UIManager.LookAndFeelInfo lafInfo;

    /**
     *  Constructor.
     *  @param info      the displayed look and feel info
     *  @param l         the locale to use
     */
    public SpecialMenuItem(@NotNull UIManager.LookAndFeelInfo info,
                           @Nullable Locale l)
    {
      super(info.getName(), info == getCurrentLookAndFeelInfo());
      setToolTipText(info.getClassName());
      addValueChangeListenerWeakly(this);
      lafInfo = info;
      addItemListener(e -> {
        if (isSelected()) {
          setLookAndFeelClass(lafInfo.getClassName());
        }
      });
    }

    /**
     * This method gets called when a bound property is changed.
     *
     * @param evt A PropertyChangeEvent object describing the event source
     *            and the property that has changed.
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt)
    {
      setSelected(lafInfo.getName() == evt.getNewValue());
    }

    /**
     * Notifies this component that it now has a parent component.
     */
    @Override
    public void addNotify()
    {
      I18n.addLocalizationChangeListener(this);
      super.addNotify();
      addValueChangeListener(this);
      if (lafInfo == getCurrentLookAndFeelInfo()) {
        setSelected(true);
      }
    }

    /**
     * Overrides {@code removeNotify} to remove this from the value listeners-
     */
    @Override
    public void removeNotify()
    {
      I18n.removeLocalizationChangeListener(this);
      super.removeNotify();
      removeValueChangeListener(this);
    }

    /**
     * Sets the locale of this component.  This is a bound property.
     *
     * @param l the locale to become this component's locale
     * @see #getLocale
     * @since JDK1.1
     */
    @Override
    public void setLocale(Locale l)
    {
      super.setLocale(l);
      setText(lafInfo.getName());
      setToolTipText(lafInfo.getClassName());
    }
  }

}
