// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.util.settings.LongPreferenceProperty;

import java.util.prefs.Preferences;

/**
 *  Editable long preference property.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingLongPreferenceProperty
        extends SwingNumberProperty
        implements LongPreferenceProperty,
                   EditablePreferenceProperty
{
  private static final long serialVersionUID = -198215960103289841L;
  /** The minimal possible value. */
  private final long minValue;
  /** The maximal possible value. */
  private final long maxValue;
  /** The step used. */
  private final long step;
  /** Current value. */
  private long value;

  /**
   *  Create a integer preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value)
  {
    this(basicName, baseTag,  value, false);
  }

  /**
   *  Create a integer preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param autoSave   save changes immediately?
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value, boolean autoSave)
  {
    this(basicName, baseTag, value, 1);
  }

  /**
   *  Create a integer preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value, long step)
  {
    this(basicName, baseTag, value, step, false);
  }

  /**
   *  Create a integer preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   *  @param autoSave   save changes immediately?
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value, long step, boolean autoSave)
  {
    this(basicName, baseTag, value, Long.MIN_VALUE, Long.MAX_VALUE, step);
  }

  /**
   *  Create a integer preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value, long minValue, long maxValue)
  {
    this(basicName, baseTag, value, minValue, maxValue, false);
  }

  /**
   *  Create a integer preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param autoSave   save changes immediately?
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value, long minValue, long maxValue, boolean autoSave)
  {
    this(basicName, baseTag, value, minValue, maxValue, 1, autoSave);
  }

  /**
   *  Create a integer preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value, long minValue, long maxValue, long step)
  {
    this(basicName, baseTag, value, minValue, maxValue, step, false);
  }

  /**
   *  Create a integer preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   *  @param autoSave   save changes immediately?
   */
  public SwingLongPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, long value, long minValue, long maxValue, long step, boolean autoSave)
  {
    super(basicName, baseTag, autoSave);
    this.minValue = minValue;
    this.maxValue = maxValue;
    this.step = step;
    this.value = value;
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    setValue(preferences.getLong(getBasicName(), value));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    preferences.putLong(getBasicName(), value);
  }

  /**
   *  Get the minimal allowed value.
   *  @return minimal value
   */
  public long getMinValue()
  {
    return minValue;
  }

  /**
   *  Get the maximal allowed value.
   *  @return maximal value
   */
  public long getMaxValue()
  {
    return maxValue;
  }

  /**
   *  Get the step used in steppers.
   *  @return step
   */
  public long getStep()
  {
    return step;
  }

  /**
   *  Get the property's value.
   *  @return property value
   */
  @Override
  @NotNull
  public Long getValue()
  {
    return value;
  }

  /**
   *  Set the property's value.
   *  @param value value to set
   */
  public void setValue(long value)
  {
    long oldValue = this.value;
    if (value > maxValue) {
      this.value = maxValue;
    }
    else if (value < minValue) {
      this.value = minValue;
    }
    else {
      this.value = value;
    }
    if (oldValue != this.value) {
      fireValueChange(getBasicName(), oldValue, this.value);
    }
  }

  /**
   * Set the property's value.
   * @param value new value
   */
  public void setValue(@NotNull Long value)
  {
    setValue(value.longValue());
  }

  /**
   * Get the minimal allowed value.
   *
   * @return minimal value
   */
  @NotNull
  @Override
  public Number getMinValueNumber()
  {
    return getMinValue();
  }

  /**
   * Get the maximal allowed value.
   *
   * @return maximal value
   */
  @NotNull
  @Override
  public Number getMaxValueNumber()
  {
    return getMaxValue();
  }

  /**
   * Get the step used in steppers.
   *
   * @return step
   */
  @NotNull
  @Override
  public Number getStepNumber()
  {
    return getStep();
  }

  /**
   * Get the properties value.
   *
   * @return property value
   */
  @NotNull
  @Override
  public Number getValueNumber()
  {
    return getValue();
  }

  /**
   * Set the property value.
   *
   * @param value new value
   */
  @Override
  public void setValueNumber(@NotNull Number value)
  {
    setValue(value.longValue());
  }
}
