// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.util.settings.Property;

/**
 *  Editable integer preference property.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingIntegerProperty
        extends SwingNumberProperty
        implements Property<Integer>
{
  private static final long serialVersionUID = 6075926052059224819L;
  /** The minimal possible value. */
  private final int minValue;
  /** The maximal possible value. */
  private final int maxValue;
  /** The step used. */
  private final int step;
  /** Current value. */
  private int value;

  /**
   *  Create a integer preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   */
  public SwingIntegerProperty(String basicName, String baseTag, int value)
  {
    this(basicName, baseTag, value, false);
  }

  /**
   *  Create a integer preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param autoSave   save changes immediately?
   */
  public SwingIntegerProperty(String basicName, String baseTag, int value, boolean autoSave)
  {
    this(basicName, baseTag, value, 1, autoSave);
  }

  /**
   *  Create a integer preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   */
  public SwingIntegerProperty(String basicName, String baseTag, int value, int step)
  {
    this(basicName, baseTag, value, step, false);
  }

  /**
   *  Create a integer preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   *  @param autoSave   save changes immediately?
   */
  public SwingIntegerProperty(String basicName, String baseTag, int value, int step, boolean autoSave)
  {
    this(basicName, baseTag, value, Integer.MIN_VALUE, Integer.MAX_VALUE, step, autoSave);
  }

  /**
   *  Create a integer preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   */
  public SwingIntegerProperty(String basicName, String baseTag, int value, int minValue, int maxValue)
  {
    this(basicName, baseTag, value, minValue, maxValue, false);
  }

  /**
   *  Create a integer preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param autoSave   save changes immediately?
   */
  public SwingIntegerProperty(String basicName, String baseTag, int value, int minValue, int maxValue, boolean autoSave)
  {
    this(basicName, baseTag, value, minValue, maxValue, 1, autoSave);
  }

  /**
   *  Create a integer preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   */
  public SwingIntegerProperty(String basicName, String baseTag, int value, int minValue, int maxValue, int step)
  {
    this(basicName, baseTag, value, minValue, maxValue, step, false);
  }

  /**
   *  Create a integer preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   *  @param autoSave   save changes immediately?
   */
  public SwingIntegerProperty(@NotNull String basicName,
                              @NotNull String baseTag, int value, int minValue, int maxValue, int step,
                              boolean autoSave)
  {
    super(basicName, baseTag, autoSave);
    this.minValue = minValue;
    this.maxValue = maxValue;
    this.step = step;
    this.value = value;
  }

  /**
   *  Get the minimal allowed value.
   *  @return minimal value
   */
  public int getMinValue()
  {
    return minValue;
  }

  /**
   *  Get the maximal allowed value.
   *  @return maximal value
   */
  public int getMaxValue()
  {
    return maxValue;
  }

  /**
   *  Get the step used in steppers.
   *  @return step
   */
  public int getStep()
  {
    return step;
  }

  /**
   *  Get the properties value.
   *  @return property value
   */
  @Override
  public Integer getValue()
  {
    return value;
  }

  /**
   * Set the value.
   *
   * @param value new value
   */
  @Override
  public void setValue(@NotNull Integer value)
  {
    setValue(value.intValue());
  }

  /**
   *  Set the properties value.
   *  @param value value to set
   */
  public void setValue(int value)
  {
    int oldValue = this.value;
    if (value > maxValue) {
      this.value = maxValue;
    }
    else if (value < minValue) {
      this.value = minValue;
    }
    else {
      this.value = value;
    }
    if (oldValue != this.value) {
      fireValueChange(getBasicName(), oldValue, this.value);
    }
  }

  /**
   * Get the minimal allowed value.
   *
   * @return minimal value
   */
  @NotNull
  @Override
  public Number getMinValueNumber()
  {
    return getMinValue();
  }

  /**
   * Get the maximal allowed value.
   *
   * @return maximal value
   */
  @NotNull
  @Override
  public Number getMaxValueNumber()
  {
    return getMaxValue();
  }

  /**
   * Get the step used in steppers.
   *
   * @return step
   */
  @NotNull
  @Override
  public Number getStepNumber()
  {
    return getStep();
  }

  /**
   * Get the properties value.
   *
   * @return property value
   */
  @NotNull
  @Override
  public Number getValueNumber()
  {
    return getValue();
  }

  /**
   * Set the property value.
   *
   * @param value new value
   */
  @Override
  public void setValueNumber(@NotNull Number value)
  {
    setValue(value.intValue());
  }
}
