// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.util.debug.Debug;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Properties;

/**
 *  Editable HTTP proxy settings.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingHttpProxyPropertySettings
        extends EditablePreferenceGroup
        implements PropertyChangeListener
{
  /** Suffix used for the HTTP port of the proxy. */
  public static final String PORT_SUFFIX = "-PORT";
  /** Suffix used for the HTTP host of the proxy. */
  public static final String HOST_SUFFIX = "-HOST";
  /** Suffix used to indicate whether the HTTP proxy is in use. */
  public static final String USED_SUFFIX = "-USED";
  /** Is a proxy used? */
  private final SwingBooleanPreferenceProperty useProxy;
  /** Proxy hostname. */
  private final SwingStringPreferenceProperty  proxyHost;
  /** Proxy port number. */
  private final SwingIntegerPreferenceProperty proxyPort;

  /**
   * Constructor.
   * @param basicName preference base name
   * @param baseTag   I18n resource base tag
   */
  public SwingHttpProxyPropertySettings(String basicName, String baseTag)
  {
    super(baseTag, false);
    useProxy = new SwingBooleanPreferenceProperty(basicName+USED_SUFFIX,
                                                  baseTag+USED_SUFFIX,
                                                  false);
    proxyHost = new SwingStringPreferenceProperty(basicName+HOST_SUFFIX,
                                                  baseTag+HOST_SUFFIX);
    proxyPort = new SwingIntegerPreferenceProperty(basicName+PORT_SUFFIX,
                                                   baseTag+PORT_SUFFIX,
                                                   80,
                                                   1, 65535);
    String pUsed = null;
    String pHost = null;
    String pPort = null;
    try {
      pUsed = System.getProperty("proxySet");
      pHost = System.getProperty("http.proxyHost");
      pPort = System.getProperty("http.proxyPort");
    } catch (SecurityException e) {
      Debug.warn(e);
      pUsed = null;  // don't use anything
    }

    if (pUsed != null && Boolean.valueOf(pUsed)) {
      useProxy.setValue(true);
      proxyHost.setValue(pHost);
      proxyPort.setValue(Integer.parseInt(pPort));
    }

    useProxy.addValueChangeListenerWeakly(this);
    proxyHost.addValueChangeListenerWeakly(this);
    proxyPort.addValueChangeListenerWeakly(this);

    addProperty(useProxy);
    addProperty(proxyHost);
    addProperty(proxyPort);
  }

  /**
   * This method gets called when a bound property is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the property that has changed.
   */

  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    if (useProxy.getValue()) {
      System.setProperty("proxySet", "true");
      final String host = proxyHost.getValue();
      System.setProperty("http.proxyHost", host != null ? host : "");
      System.setProperty("http.proxyPort", Integer.toString(proxyPort.getValue()));
    }
    else {
      Properties properties = System.getProperties();
      properties.remove("http.proxyHost");
      properties.remove("http.proxyPort");
      properties.remove("proxySet");
      System.setProperties(properties);
    }
  }
}
