// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.settings.Property;

import javax.swing.*;
import java.util.Locale;

/**
 * Property represented as a one-line string with a defined format.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingFormattedStringProperty<T>
        extends AbstractBasicSimpleEditableChangeableItem
        implements Property<T>
{
  /** Default number of rows displayed in editor. */
  public static final int DEFAULT_ROWS = 40;
  private static final long serialVersionUID = -1880148426126237161L;
  /** The value. */
  protected T value;
  /** Auto save? */
  private final boolean autoSave;
  /** Columns to display. */
  private final int columns;
  /** Formatter to use. */
  @NotNull
  private final JFormattedTextField.AbstractFormatter formatter;

  /**
   * Constructor.
   * @param basicName  basic name
   * @param baseTag    base tag
   * @param formatter  formatter defining the expected format of the string
   */
  public SwingFormattedStringProperty(@NotNull String basicName,
                                      @NotNull String baseTag,
                                      @NotNull JFormattedTextField.AbstractFormatter formatter)
  {
    this(basicName, baseTag, formatter, false, null, DEFAULT_ROWS);
  }

  /**
   * Constructor.
   * @param basicName  basic name
   * @param baseTag    base tag
   * @param formatter  formatter defining the expected format of the string
   * @param autoSave   do auto save?
   * @param value      initial value
   * @param columns    columns to display
   */
  public SwingFormattedStringProperty(@NotNull String basicName,
                                      @NotNull String baseTag,
                                      @NotNull JFormattedTextField.AbstractFormatter formatter,
                                      boolean autoSave,
                                      T value,
                                      int columns)
  {
    super(basicName, baseTag);
    this.autoSave = autoSave;
    this.value = value;
    this.formatter = formatter;
    this.columns = columns;
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor component
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new FormattedStringEditor<>(this, autoSave, l);
  }

  /**
   * Get the number of columns to be displayed.
   * @return number of displayed columns
   */
  public int getColumns()
  {
    return columns;
  }

  /**
   *  Get the value.
   *  @return value
   */
  @Override
  public T getValue()
  {
    return value;
  }

  /**
   * Get the formatter to use.
   * @return formatter
   */
  @NotNull
  public JFormattedTextField.AbstractFormatter getFormatter()
  {
    return formatter;
  }

  /**
   *  Set the value.
   *  @param value  string value
   */
  @Override
  public void setValue(T value)
  {
    if (value != null ? !value.equals(this.value) : this.value != null) {
      T oldValue = this.value;
      this.value = value;
      fireValueChange(getBasicName(), oldValue, value);
    }
  }
}
