// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.i18n.swing.RJMenu;
import de.caff.util.settings.EnumProperty;
import de.caff.util.settings.EnumValue;

import javax.swing.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.*;

/**
 *  An enum preference property.
 *  This is one value out of a set of values.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingEnumProperty<E>
        extends AbstractBasicSimpleEditableChangeableItem
        implements EnumProperty<E>
{
  private static final long serialVersionUID = -910957808821606537L;
  /** The possible values. */
  private final List<EnumValue<E>> values;
  /** The currently selected value. */
  private EnumValue<E> selectedValue;
  /** The comparator, possibly {@code null}. */
  private EnumValueComparator<E> enumValueComparator;
  /** Auto save values? */
  private final boolean autoSave;

  /**
   *  Constructor.
   *  No auto saving.
   *
   * @param startValue the start value of this property
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param values     the possible values of this property
   */
  @SuppressWarnings("unchecked")
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull EnumValue<E>... values)
  {
    this(startValue, basicName, baseTag, Types.asList(values));
  }

  /**
   *  Constructor.
   *
   * @param startValue the start value of this property
   * @param autoSave   auto save changes?
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param values     the possible values of this property
   */
  @SuppressWarnings("unchecked")
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           boolean autoSave,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull EnumValue<E>... values)
  {
    this(startValue, autoSave, basicName, baseTag, Types.asList(values));
  }

  /**
   * Constructor.
   * No auto saving.
   *
   * @param startValue the start value of this property
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param values     the possible values of this property
   */
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull Collection<EnumValue<E>> values)
  {
    this(startValue, false, basicName, baseTag, values);
  }

  /**
   * Constructor.
   *
   * @param startValue the start value of this property
   * @param autoSave   auto save changes?
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param values     the possible values of this property
   */
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           boolean autoSave,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull Collection<EnumValue<E>> values)
  {
    super(basicName, baseTag);
    this.values = new ArrayList<>(values);
    this.selectedValue = startValue;
    this.autoSave = autoSave;
  }

  /**
   * Constructor.
   * No auto save.
   *
   * @param startValue the start value of this property
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param enumValueComparator comparator for {@link de.caff.util.settings.EnumValue}s used to sort the values for display
   * @param values     the possible values of this property
   */
  @SuppressWarnings("unchecked")
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull EnumValueComparator<E> enumValueComparator,
                           @NotNull EnumValue<E>... values)
  {
    this(startValue, basicName, baseTag, values);
    this.enumValueComparator = enumValueComparator;
  }

  /**
   * Constructor.
   *
   * @param startValue the start value of this property
   * @param autoSave   auto save values?
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param enumValueComparator comparator for {@link de.caff.util.settings.EnumValue}s used to sort the values for display
   * @param values     the possible values of this property
   */
  @SuppressWarnings("unchecked")
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           boolean autoSave,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull EnumValueComparator<E> enumValueComparator,
                           @NotNull EnumValue<E>... values)
  {
    this(startValue, autoSave, basicName, baseTag, values);
    this.enumValueComparator = enumValueComparator;
  }

  /**
   * Constructor.
   *
   * @param startValue the start value of this property
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param enumValueComparator comparator for {@link de.caff.util.settings.EnumValue}s used to sort the values for display
   * @param values     the possible values of this property
   */
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull EnumValueComparator<E> enumValueComparator,
                           @NotNull Collection<EnumValue<E>> values)
  {
    this(startValue, basicName, baseTag, values);
    this.enumValueComparator = enumValueComparator;
  }

  /**
   *  Constructor.
   *
   * @param startValue the start value of this property
   * @param autoSave   auto save values?
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param enumValueComparator comparator for {@link de.caff.util.settings.EnumValue}s used to sort the values for display
   * @param values     the possible values of this property
   */
  public SwingEnumProperty(@NotNull EnumValue<E> startValue,
                           boolean autoSave,
                           @NotNull String basicName,
                           @NotNull String baseTag,
                           @NotNull EnumValueComparator<E> enumValueComparator,
                           @NotNull Collection<EnumValue<E>> values)
  {
    this(startValue, autoSave, basicName, baseTag, values);
    this.enumValueComparator = enumValueComparator;
  }

  /**
   * Get the editor provider which provides editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor provider
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new EnumEditor<>(this, autoSave, l);
  }

  /**
   *  Get the current value.
   *  @return the value
   */
  @NotNull
  @Override
  public EnumValue<E> getValue()
  {
    return selectedValue;
  }

  /**
   *  Set a new value.
   *  @param value new value
   */
  public void setValue(EnumValue<E> value)
  {
    if (!value.equals(selectedValue)) {
      EnumValue<E> oldValue = selectedValue;
      selectedValue = value;
      fireValueChange(getBasicName(), oldValue, value);
    }
  }

  /**
   *  Get the possible values.
   *  @return possible values
   */
  @NotNull
  @Override
  public List<EnumValue<E>> getEnumValues(Locale l)
  {
    return getEnumValues();
  }

  /**
   * Get the enum values.
   * @return enum values
   */
  @NotNull
  protected List<EnumValue<E>> getEnumValues()
  {
    return Collections.unmodifiableList(values);
  }

  /**
   * Get the enum value comparator.
   * @return enum value comparator used  to sort the enum values for display,
   *         or {@code null} if they are displayed in their natural order
   */
  EnumValueComparator<E> getEnumValueComparator()
  {
    return enumValueComparator;
  }

  private class SpecialMenuItem
          extends JRadioButtonMenuItem
          implements PropertyChangeListener,
                     Localizable
  {
    private static final long serialVersionUID = 3230453897448041768L;
    /** The value. */
    private final EnumValue<E> value;

    /**
     *  Constructor.
     *  @param enumValue the enum value to display
     *  @param l         the locale to use
     */
    public SpecialMenuItem(EnumValue<E> enumValue, Locale l)
    {
      super(enumValue.getShortName(l), enumValue == getValue());
      setToolTipText(enumValue.getLongName(l));
      addValueChangeListenerWeakly(this);
      this.value = enumValue;
      addItemListener(e -> {
        if (isSelected()) {
          setValue(value);
        }
      });
    }

    /**
     * This method gets called when a bound property is changed.
     *
     * @param evt A PropertyChangeEvent object describing the event source
     *            and the property that has changed.
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt)
    {
      setSelected(value == evt.getNewValue());
    }

    /**
     * Notifies this component that it now has a parent component.
     */
    @Override
    public void addNotify()
    {
      I18n.addLocalizationChangeListener(this);
      super.addNotify();
      addValueChangeListener(this);
      if (value == getValue()) {
        setSelected(true);
      }
    }

    /**
     * Overrides {@code removeNotify} to remove this from the value listeners-
     */
    @Override
    public void removeNotify()
    {
      I18n.removeLocalizationChangeListener(this);
      super.removeNotify();
      removeValueChangeListener(this);
    }

    /**
     * Sets the locale of this component.  This is a bound property.
     *
     * @param l the locale to become this component's locale
     * @see #getLocale
     * @since JDK1.1
     */
    @Override
    public void setLocale(Locale l)
    {
      super.setLocale(l);
      setText(value.getShortName(l));
      setToolTipText(value.getLongName(l));
    }
  }

  /**
   *  Create a menu item for this boolean property.
   *  @param l locale
   *  @return menu item
   */
  public JMenuItem createMenuItem(Locale l)
  {
    RJMenu menu = new RJMenu(getBaseTag());
    ButtonGroup group = new ButtonGroup();
    for (EnumValue<E> value: values) {
      SpecialMenuItem menuItem = new SpecialMenuItem(value, l);
      menu.add(menuItem);
      group.add(menuItem);
    }
    return menu;
  }

  /**
   * Get an enum value comparator suitable to be used in the
   * {@link de.caff.util.settings.swing.SwingEnumProperty#SwingEnumProperty(de.caff.util.settings.EnumValue, String, String, de.caff.util.settings.swing.SwingEnumProperty.EnumValueComparator, de.caff.util.settings.EnumValue[])}
   * constructor, which sorts the enum values by their short names.
   * @param <T> enum type
   * @return comparator comparing short names (case insensitive)
   */
  public static <T> EnumValueComparator<T> getShortNameEnumValueComparator()
  {
    return new EnumValueComparator<T>()
    {
      private Locale locale;

      @Override
      public void setLocale(Locale l)
      {
        locale = l;
      }

      @Override
      public int compare(EnumValue<T> v1, EnumValue<T> v2)
      {
        return v1.getShortName(locale).compareToIgnoreCase(v2.getShortName(locale));
      }
    };
  }

  /**
   * Comparator used for enum value sorting.
   */
  public static interface EnumValueComparator<T>
    extends Comparator<EnumValue<T>>
  {
    /**
     *  Set the locale this comparator uses.
     *  @param  l   locale to set.
     */
    public void setLocale(Locale l);
  }
}
