// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.util.settings.EnumPreferenceProperty;
import de.caff.util.settings.EnumValue;

import java.util.Collection;
import java.util.prefs.Preferences;

/**
 *  An enum preference property.
 *  This is one value out of a set of values.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingEnumPreferenceProperty<E>
        extends SwingEnumProperty<E>
        implements EnumPreferenceProperty<E>,
                   EditablePreferenceProperty
{
  private static final long serialVersionUID = -2052290866442352872L;

  /**
   *  Constructor.
   *
   * @param startValue the start value of this property
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param values     the possible values of this property
   */
  @SuppressWarnings("unchecked")
  public SwingEnumPreferenceProperty(@NotNull EnumValue<E> startValue,
                                     @NotNull String basicName,
                                     @NotNull String baseTag,
                                     @NotNull EnumValue<E> ... values)
  {
    super(startValue, basicName, baseTag, values);
  }

  /**
   * Constructor.
   *
   * @param startValue the start value of this property
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param values     the possible values of this property
   */
  public SwingEnumPreferenceProperty(@NotNull EnumValue<E> startValue,
                                     @NotNull String basicName,
                                     @NotNull String baseTag,
                                     @NotNull Collection<EnumValue<E>> values)
  {
    super(startValue, basicName, baseTag, values);
  }

  /**
   * Constructor.
   *
   * @param startValue the start value of this property
   * @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param enumValueComparator comparator for {@link de.caff.util.settings.EnumValue}s used to sort the values for display
   * @param values     the possible values of this property
   */
  @SuppressWarnings("unchecked")
  public SwingEnumPreferenceProperty(@NotNull EnumValue<E> startValue,
                                     @NotNull String basicName,
                                     @NotNull String baseTag,
                                     @NotNull EnumValueComparator<E> enumValueComparator,
                                     @NotNull EnumValue<E> ... values)
  {
    super(startValue, basicName, baseTag, enumValueComparator, values);
  }

  /**
   *  Constructor.
   *
   * @param startValue the start value of this property
   *  @param basicName  the name of this property
   * @param baseTag    the basic i18n tag of this property
   * @param enumValueComparator comparator for {@link de.caff.util.settings.EnumValue}s used to sort the values for display
   * @param values     the possible values of this property
   */
  public SwingEnumPreferenceProperty(@NotNull EnumValue<E> startValue,
                                     @NotNull String basicName,
                                     @NotNull String baseTag,
                                     @NotNull EnumValueComparator<E> enumValueComparator,
                                     @NotNull Collection<EnumValue<E>> values)
  {
    super(startValue, basicName, baseTag, enumValueComparator, values);
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    for (EnumValue<E> value: getEnumValues()) {
      if (value.isEqualTo(preferences, getBasicName())) {
        setValue(value);
        break;
      }
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    getValue().storeTo(preferences, getBasicName());
  }

}
