// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.util.settings.DoublePreferenceProperty;

import java.util.prefs.Preferences;

/**
 *  Editable double preference property.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingDoublePreferenceProperty
        extends SwingDoubleProperty
        implements DoublePreferenceProperty,
                   EditablePreferenceProperty
{
  private static final long serialVersionUID = -635152081189321398L;

  /**
   *  Create a double preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag, double value)
  {
    this(basicName, baseTag, value, false);
  }

  /**
   *  Create a double preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param autoSave   save changes immediately?
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag, double value, boolean autoSave)
  {
    this(basicName, baseTag, value, 1, autoSave);
  }

  /**
   *  Create a double preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag, double value, double step)
  {
    this(basicName, baseTag, value, step, false);
  }

  /**
   *  Create a double preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   *  @param autoSave   save changes immediately?
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag, double value, double step, boolean autoSave)
  {
    this(basicName, baseTag, value, -Double.MAX_VALUE, Double.MAX_VALUE, step, autoSave);
  }

  /**
   *  Create a double preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag, double value, double minValue, double maxValue)
  {
    this(basicName, baseTag, value, minValue, maxValue, false);
  }

  /**
   *  Create a double preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param autoSave   save changes immediately?
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag, double value, double minValue, double maxValue, boolean autoSave)
  {
    this(basicName, baseTag, value, minValue, maxValue, 1, autoSave);
  }

  /**
   *  Create a double preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag,
                                       double value, double minValue, double maxValue, double step)
  {
    this(basicName, baseTag, value, minValue, maxValue, step, false);
  }

  /**
   *  Create a double preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   *  @param autoSave   save changes immediately?
   */
  public SwingDoublePreferenceProperty(@NotNull String basicName, @NotNull String baseTag,
                                       double value, double minValue, double maxValue, double step,
                                       boolean autoSave)
  {
    super(basicName, baseTag, value, minValue, maxValue, step, autoSave);
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    setValue(preferences.getDouble(getBasicName(), getValue()));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    preferences.putDouble(getBasicName(), getValue());
  }

}
