// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.settings.DirectoryPreferenceProperty;

import java.io.File;
import java.util.Locale;
import java.util.prefs.Preferences;

/**
 *  Editable directory preference property.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingDirectoryPreferenceProperty
        extends AbstractBasicSimpleEditableChangeableItem
        implements DirectoryPreferenceProperty,
                   EditablePreferenceProperty
{
  private static final long serialVersionUID = -3313212683306473948L;
  /** The directory. */
  private File directory;

  /**
   *  Constructor.
   *  @param basicName  basic name (prefix)
   *  @param baseTag    basic i18n tag
   *  @param directory  default value
   */
  public SwingDirectoryPreferenceProperty(@NotNull String basicName,
                                          @NotNull String baseTag,
                                          @Nullable File directory)
  {
    super(basicName, baseTag);
    this.directory = directory;
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    String value = preferences.get(getBasicName(), null);
    if (value != null) {
      setDirectory(new File(value));
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    if (directory == null) {
      preferences.remove(getBasicName());
    }
    else {
      preferences.put(getBasicName(), directory.getPath());
    }
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor components
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new DirectoryEditor(this, l);
  }

  /**
   *  Get the directory.
   *  @return the directory
   */
  @Override
  public File getDirectory()
  {
    return directory;
  }

  /**
   *  Set the directory.
   *  @param directory  the directory
   */
  @Override
  public void setDirectory(File directory)
  {
    if (directory != null ? !directory.equals(this.directory) : this.directory != null) {
      File oldDirectory = this.directory;
      this.directory = directory;
      fireValueChange(getBasicName(), oldDirectory, directory);
    }
  }
}
