// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.swing.RJMenuItem;
import de.caff.util.settings.Property;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Locale;

/**
 * Color property.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingColorProperty
        extends AbstractBasicSimpleEditableChangeableItem
        implements Property<Color>
{
  private static final long serialVersionUID = -375223870683758152L;
  /** Save values automatically? */
  protected final boolean autoSave;
  /** The color. */
  protected Color value;

  /**
   * Constructor.
   * @param basicName       basic property name
   * @param baseTag         basic I18n tag
   * @param autoSave        do auto save?
   * @param defaultValue    default value
   */
  public SwingColorProperty(@NotNull String basicName,
                            @NotNull String baseTag,
                            boolean autoSave,
                            @NotNull Color defaultValue)
  {
    super(basicName, baseTag);
    this.autoSave = autoSave;
    value = defaultValue;
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor component
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new ColorEditor(this, l, autoSave);
  }

  /**
   *  Get the color value,.
   *  @return color value
   */
  @Override
  public Color getValue()
  {
    return value;
  }

  /**
   *  Set the color value.
   *  @param value new color
   */
  public void setValue(Color value)
  {
    if (value != null ? !value.equals(this.value) : this.value != null) {
      Color oldColor = this.value;
      this.value = value;
      fireValueChange(getBasicName(), oldColor, value);
    }
  }

  /**
   *  Create a menu entry which allows to switch to a given color.
   *  @param tag    i18n tag to use for switching to color
   *  @param color  color to switch to
   *  @param l      locale to use
   *  @return menu item
   */
  public JMenuItem createMenuItem(@NotNull String tag,
                                  @NotNull Color color,
                                  @Nullable Locale l)
  {
    return new SpecialMenuItem(tag, color, l);
  }

  private class SpecialMenuItem
          extends RJMenuItem
          implements PropertyChangeListener
  {
    private static final long serialVersionUID = 5617140937616415191L;
    /** The color. */
    private final Color color;

    /**
     *  Constructor.
     *  @param tag basic resource tag
     *  @param color start color
     *  @param l     current locale
     */
    public SpecialMenuItem(String tag, Color color, Locale l)
    {
      super(tag, l);
      this.color = color;
      setIcon(ColorEditor.createColoredIcon(color));

      addActionListener(e -> setValue(SpecialMenuItem.this.color));
    }

    /**
     * This method gets called when a bound property is changed.
     *
     * @param evt A PropertyChangeEvent object describing the event source
     *            and the property that has changed.
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt)
    {
      final Color newColor = (Color)evt.getNewValue();
      handleNewColor(newColor);
    }

    private void handleNewColor(final Color newColor)
    {
      setEnabled(!color.equals(newColor));
    }

    /**
     * Notifies this component that it now has a parent component.
     */
    @Override
    public void addNotify()
    {
      super.addNotify();
      addValueChangeListener(this);
      handleNewColor(getValue());
    }

    /**
     * Overrides {@code removeNotify} to remove this from the value listeners-
     */
    @Override
    public void removeNotify()
    {
      super.removeNotify();
      removeValueChangeListener(this);
    }
  }
}
