// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.swing.RJCheckBoxMenuItem;
import de.caff.util.settings.Property;

import javax.swing.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Locale;

/**
 *  Editable boolean preference property.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingBooleanProperty
        extends AbstractBasicSimpleEditableChangeableItem
        implements Property<Boolean>
{
  private static final long serialVersionUID = 6793470999877380631L;
  /** Automatically save value? */
  private final boolean autoSave;
  /** The current value. */
  private boolean value;


  /**
   * Constructor.
   * @param basicName  property tag
   * @param baseTag    i18n tag
   * @param autoSave   automatically save value on changes?
   * @param value      default value
   */
  public SwingBooleanProperty(@NotNull String basicName, @NotNull String baseTag,
                              boolean autoSave, boolean value)
  {
    super(basicName, baseTag);
    this.autoSave = autoSave;
    this.value = value;
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor components
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new BooleanEditor(this, l);
  }

  /**
   * Is auto save enabled?
   * @return the answer
   */
  public boolean isAutoSave()
  {
    return autoSave;
  }

  /**
   *  Get the current value.
   *  @return the value
   */
  @Override
  public Boolean getValue()
  {
    return value;
  }

  /**
   * Set the value.
   *
   * @param value new value
   */
  @Override
  public void setValue(@NotNull Boolean value)
  {
    setValue(value.booleanValue());
  }

  /**
   *  Set the value.
   *  @param value new value
   */
  public void setValue(boolean value)
  {
    if (this.value != value) {
      this.value = value;
      fireValueChange(getBasicName(), !value, value);
    }
  }

  /**
   * Menu item with a checkbox.
   */
  private class SpecialMenuItem
          extends RJCheckBoxMenuItem
          implements PropertyChangeListener
  {
    private static final long serialVersionUID = 3003396281177599035L;

    /**
     *  Constructor.
     */
    public SpecialMenuItem()
    {
      super(getBaseTag(), getValue());
      addItemListener(e -> setValue(isSelected()));
    }

    /**
     * This method gets called when a bound property is changed.
     *
     * @param evt A PropertyChangeEvent object describing the event source
     *            and the property that has changed.
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt)
    {
      setSelected(((Boolean)evt.getNewValue()).booleanValue());
    }

    /**
     * Notifies this component that it now has a parent component.
     */
    @Override
    public void addNotify()
    {
      super.addNotify();
      addValueChangeListener(this);
      setSelected(getValue());
    }

    /**
     * Overrides {@code removeNotify} to remove this from the value listeners-
     */
    @Override
    public void removeNotify()
    {
      super.removeNotify();
      removeValueChangeListener(this);
    }
  }

  /**
   *  Create a menu item for this boolean property.
   *  @param l locale
   *  @return menu item
   */
  public JMenuItem createMenuItem(Locale l)
  {
    SpecialMenuItem menuItem = new SpecialMenuItem();
    menuItem.setLocale(l);
    return menuItem;
  }
}
