// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.util.settings.BooleanPreferenceProperty;

import java.util.prefs.Preferences;

/**
 *  Editable boolean preference property.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SwingBooleanPreferenceProperty
        extends SwingBooleanProperty
        implements BooleanPreferenceProperty,
                   EditablePreferenceProperty
{
  private static final long serialVersionUID = 6793470999877380631L;

  /**
   *  Constructor.
   *  @param basicName  property tag
   *  @param baseTag    i18n tag
   *  @param value      default value
   */
  public SwingBooleanPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, boolean value)
  {
    this(basicName, baseTag, false, value);
  }

  /**
   *  Constructor.
   *  @param basicName  property tag
   *  @param baseTag    i18n tag
   *  @param autoSave   automatically save value on changes?
   *  @param value      default value
   */
  public SwingBooleanPreferenceProperty(@NotNull String basicName, @NotNull String baseTag, boolean autoSave, boolean value)
  {
    super(basicName, baseTag, autoSave, value);
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    setValue(preferences.getBoolean(getBasicName(), getValue()));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    preferences.putBoolean(getBasicName(), getValue());
  }
}
