// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.swing.RJButton;
import de.caff.util.settings.BoundsPreferenceProperty;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;

/**
 *  A dialog allowing to change certain settings.
 *  
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class SettingsDialog
        extends JDialog
{
  private static final long serialVersionUID = 8834918186045917419L;
  /** The editor provider. */
  private final EditorProvider editorProvider;

  /**
   *  Constructor.
   *  @param component component of the window for which this dialog is displayed
   *  @param title     dialog title
   *  @param bpp       preference for bounds
   *  @param editorProvider editor provider
   */
  public SettingsDialog(@NotNull JComponent     component,
                        @NotNull String         title,
                        @NotNull final BoundsPreferenceProperty bpp,
                        @NotNull EditorProvider editorProvider)
  {
    super(getFrameAncestor(component), title, true);
    this.editorProvider = editorProvider;
    getContentPane().setLayout(new BorderLayout());
    getContentPane().add(editorProvider.getEditor(), BorderLayout.CENTER);

    JButton ok     = new RJButton("Ok");
    JButton cancel = new RJButton("Cancel");
    Box buttons = Box.createHorizontalBox();
    buttons.add(Box.createHorizontalGlue());
    buttons.add(ok);
    buttons.add(cancel);
    buttons.add(Box.createHorizontalGlue());
    getContentPane().add(buttons, BorderLayout.SOUTH);

    ok.addActionListener(e -> ok(bpp));
    cancel.addActionListener(e -> cancel(bpp));
    addWindowListener(new WindowAdapter()
    {
      @Override
      public void windowClosing(WindowEvent e)
      {
        cancel(bpp);
      }
    });
    setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
    if (!bpp.setWindowBounds(this)) {
      pack();
      setLocationRelativeTo(component);
    }
  }

  /**
   *  Get a frame ancestor for a given component.
   *  @param comp component
   *  @return frame in which this component is displayed or {@code null} if the component is not
   *          displayed in a frame
   */
  @Nullable
  private static Frame getFrameAncestor(@NotNull JComponent comp)
  {
    Window window = SwingUtilities.getWindowAncestor(comp);
    if (window instanceof Frame) {
      return (Frame)window;
    }
    else {
      return null;
    }
  }

  /**
   *  Called when the Ok button is pressed.
   *  @param bpp       preference for bounds
   */
  private void ok(@NotNull BoundsPreferenceProperty bpp)
  {
    editorProvider.save();
    dispose(bpp);
  }

  /**
   *  Called when the dialog is canceled.
   *  @param bpp       preference for bounds
   */
  private void cancel(@NotNull BoundsPreferenceProperty bpp)
  {
    dispose(bpp);
  }

  /**
   *  Disposes the Dialog and then causes show() to return if it is currently
   *  blocked.
   *  @param bpp       preference for bounds
   */
  public void dispose(@NotNull BoundsPreferenceProperty bpp)
  {
    bpp.setBounds(this);
    editorProvider.goodBye();

    super.dispose();
  }
}
