// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import javax.swing.*;
import java.awt.*;
import java.util.Iterator;
import java.util.Locale;

/**
 *  A editor provider for a preference group which displays all information in one panel.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class PanelGroupEditor
        extends AbstractBasicGroupEditor
{
  /** The panel which holds the properties. */
  private final JPanel panel;

  /**
   *  Constructor.
   *  @param property group property
   *  @param l        locale
   */
  public PanelGroupEditor(@NotNull EditablePropertyGroup<? extends EditableProperty> property,
                          @Nullable Locale l)
  {
    super(property.getNameTag(), l);
    GridBagLayout gbl = new GridBagLayout();
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.ipadx = 4;
    constraints.ipady = 4;
    constraints.insets = new Insets(6, 4, 6, 4);
    constraints.gridy = 0;

    panel = new JPanel(gbl);
    for (Iterator<? extends EditableProperty> iterator = property.propertyIterator();  iterator.hasNext();  ) {
      EditableProperty epp = iterator.next();
      EditorProvider provider = epp.getEditorProvider(l);
      addEditorProvider(provider);

      constraints.gridx = 0;
      final JComponent label = provider.getLabel();
      if (label != null) {
        constraints.gridwidth = 1;
        constraints.fill      = GridBagConstraints.HORIZONTAL;
        constraints.anchor    = GridBagConstraints.NORTHWEST;
        gbl.setConstraints(label, constraints);
        panel.add(label);
        label.setToolTipText(provider.getToolTipText());

        ++constraints.gridx;
        constraints.anchor    = GridBagConstraints.NORTHEAST;
        constraints.gridwidth = GridBagConstraints.REMAINDER;
      }
      else {
        constraints.anchor    = GridBagConstraints.NORTH;
        constraints.gridwidth = 2;
      }
      constraints.fill      = GridBagConstraints.BOTH;
      JComponent editor = provider.getEditor();
      gbl.setConstraints(editor, constraints);
      panel.add(editor);
      editor.setToolTipText(provider.getToolTipText());
      if (label == null) {
        String title = provider.getTitleText();
        if (title != null) {
          editor.setBorder(BorderFactory.createTitledBorder(title));
        }
      }
      ++constraints.gridy;
    }
  }

  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return panel;
  }

}
