// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.debug.Debug;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.NumberFormat;
import java.util.Locale;

/**
 *  Basic editor for JSpinner based editor.
 * <p>
 *  This uses a JSpinner to change the value.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class NumberEditor
        extends AbstractBasicEditorProvider
        implements PropertyChangeListener
{
  /** Often spinners have strange preferred and minimum widths. This is the workaround fix value. */
  private static final int PREFERRED_WIDTH = 60;
  private static final int MINIMUM_WIDTH = 24;

  /** Spinner used to display and set the property. */
  @Nullable
  protected final JSpinner spinner;
  /** Field used to display and set the property. */
  @Nullable
  protected final JFormattedTextField field;
  /** The editor, either the spinner or the field. */
  @NotNull
  protected final JComponent editor;
  /** The property holding the number. */
  @NotNull
  protected final SwingNumberProperty propertySwing;

  /**
   *  Constructor.
   *  @param propertySwing property to handle
   *  @param l current locale
   *  @param autoSave save changes immediately?
   */
  public NumberEditor(@NotNull SwingNumberProperty propertySwing, Locale l, boolean autoSave)
  {
    super(propertySwing, l);
    this.propertySwing = propertySwing;
    if (propertySwing.getStepNumber().doubleValue() < 0) {
      spinner = null;
      NumberFormat format = NumberFormat.getNumberInstance();
      field = new JFormattedTextField(format);
      field.setValue(propertySwing.getValueNumber());
      Dimension size = field.getPreferredSize();
      field.setPreferredSize(new Dimension(PREFERRED_WIDTH, size.height));
      editor = field;
      if (autoSave) {
        field.getDocument().addDocumentListener(new DocumentListener()
        {
          private void possiblySave()
          {
            if (field.isValid()) {
              save();
            }
          }

          @Override
          public void insertUpdate(DocumentEvent e)
          {
            possiblySave();
          }

          @Override
          public void removeUpdate(DocumentEvent e)
          {
            possiblySave();
          }

          @Override
          public void changedUpdate(DocumentEvent e)
          {
            possiblySave();
          }
        });
      }
    }
    else {
      spinner = new JSpinner(new SpinnerNumberModel(propertySwing.getValueNumber(),
                                                    (Comparable)propertySwing.getMinValueNumber(),
                                                    (Comparable)propertySwing.getMaxValueNumber(),
                                                    propertySwing.getStepNumber()));
      field = null;
      editor = spinner;
      // workaround for spinner being extremely wide (4541 pixels!)
      Dimension size = spinner.getPreferredSize();
      if (size.width > PREFERRED_WIDTH) {
        spinner.setPreferredSize(new Dimension(PREFERRED_WIDTH, size.height));
        spinner.setMinimumSize(new Dimension(MINIMUM_WIDTH, size.height));
      }
      if (autoSave) {
        spinner.addChangeListener(e -> save());
      }
    }
    propertySwing.addValueChangeListenerWeakly(this);
  }

  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return editor;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    if (spinner != null) {
      spinner.setValue(propertySwing.getValueNumber());
    }
    if (field != null) {
      field.setValue(propertySwing.getValueNumber());
    }
  }

  /**
   * Get the editor value.
   * @return editor value
   */
  private Number getEditorValue()
  {
    if (spinner != null) {
      return (Number)spinner.getValue();
    }
    assert field != null;
    return (Number)field.getValue();
  }

  /**
   * Set the editor value.
   * @param value new value
   */
  private void setEditorValue(Object value)
  {
    if (spinner != null) {
      spinner.setValue(value);

    }
    else {
      assert field != null;
      try {
        field.setValue(value);
      } catch (IllegalStateException e) {
        Debug.trace(e);
      }
    }
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    propertySwing.setValueNumber(getEditorValue());
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
    propertySwing.removeValueChangeListener(this);
  }

  /**
   * This method gets called when a bound propertySwing is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the propertySwing that has changed.
   */

  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    Object newValue = evt.getNewValue();
    if (!newValue.equals(getEditorValue())) {
      setEditorValue(newValue);
    }
  }
}
