// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.settings.EnumValue;

import javax.swing.*;
import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Locale;
import java.util.prefs.Preferences;

/**
 *  An editor for enum values.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class EnumEditor<E>
        extends AbstractBasicEditorProvider
        implements PropertyChangeListener
{
  /**
   *  Internal class used for localization.
   */
  private static class EnumValueLocalizer<F>
          implements EnumValue<F>
  {
    /** The wrapped enum value. */
    private final EnumValue<F> wrapped;
    /** The locale to use for localizatrion. */
    private final Locale locale;

    public EnumValueLocalizer(@NotNull EnumValue<F> wrapped, @Nullable Locale locale)
    {
      this.wrapped = wrapped;
      this.locale = locale;
    }

    @NotNull
    @Override
    public String getShortName(@Nullable Locale l)
    {
      return wrapped.getShortName(l);
    }

    @Override
    public String getLongName(@Nullable Locale l)
    {
      return wrapped.getLongName(l);
    }

    /**
     * Get a possible icon to display for this value.
     *
     * @param l locale to use
     * @return icon or {@code null} if no icon is available
     */
    @Override
    public Icon getIcon(@Nullable Locale l)
    {
      return wrapped.getIcon(l);
    }

    @Override
    public void storeTo(@NotNull Preferences preferences, @NotNull String key)
    {
      wrapped.storeTo(preferences, key);
    }

    @Override
    public boolean isEqualTo(@NotNull Preferences preferences, @NotNull String key)
    {
      return wrapped.isEqualTo(preferences, key);
    }

    @Override
    public F getRealValue()
    {
      return wrapped.getRealValue();
    }

    public EnumValue<F> getBasicValue()
    {
      return wrapped;
    }

    @Override
    public String toString()
    {
      return wrapped.getShortName(locale);
    }
  }

  /** The preference property which is set by this editor. */
  private final SwingEnumProperty<E> property;

  /** The combo box displaying the values. */
  @SuppressWarnings("rawtypes") // necessary for Java 7
  private JComboBox<EnumValueLocalizer<E>> comboBox;

  /**
   * Constructor.
   * @param property edited property
   * @param autoSave automatically save changed values?
   * @param l        locale
   */
  @SuppressWarnings({"rawtypes", "unchecked"}) // necessary for Java 7 JComboBox
  public EnumEditor(@NotNull SwingEnumProperty<E> property, boolean autoSave, final Locale l)
  {
    super(property, l);
    this.property = property;

    java.util.List<EnumValue<E>> enumValues = new ArrayList<>(property.getEnumValues(l));
    final SwingEnumProperty.EnumValueComparator<E> enumValueComparator = property.getEnumValueComparator();
    if (enumValueComparator != null) {
      // sort the values for display
      enumValueComparator.setLocale(l);
      Collections.sort(enumValues, enumValueComparator);
    }
    final DefaultComboBoxModel model = new DefaultComboBoxModel();
    final EnumValue value = property.getValue();
    int selected = enumValues.size() - 1;
    for (int e = 0;  e < enumValues.size();  ++e) {
      if (enumValues.get(e) == value) {
        selected = e;
      }
      model.addElement(new EnumValueLocalizer<>(enumValues.get(e), l));
    }
    comboBox = new JComboBox<>(model);
    if (selected >= 0) {
      comboBox.setSelectedIndex(selected);
    }
    comboBox.setRenderer(new DefaultListCellRenderer() {
      private static final long serialVersionUID = 3087615301542781717L;

      @Override
      public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected,
                                                    boolean cellHasFocus)
      {
        JLabel comp =  (JLabel)super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
        if (value != null) {
          final EnumValue enumValue = (EnumValue)value;
          comp.setToolTipText(enumValue.getLongName(l));
          setIcon(enumValue.getIcon(l));
        }
        return comp;
      }
    });
    if (autoSave) {
      comboBox.addItemListener(e -> save());
    }
    property.addValueChangeListenerWeakly(this);
  }

  @Override
  @SuppressWarnings("unchecked")
  public void propertyChange(PropertyChangeEvent evt)
  {
    DefaultComboBoxModel<EnumValueLocalizer<E>> model = (DefaultComboBoxModel<EnumValueLocalizer<E>>)comboBox.getModel();
    Object newValue = evt.getNewValue();
    for (int e = 0;  e < model.getSize();  ++e) {
      EnumValueLocalizer<E> value = model.getElementAt(e);
      if (value.getBasicValue().equals(newValue)) {
        comboBox.setSelectedIndex(e);
        break;
      }
    }
  }

  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return comboBox;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  @SuppressWarnings({ "unchecked", "rawtypes" }) // necessary for Java 7 JComboBox
  public void reset()
  {
    DefaultComboBoxModel model = (DefaultComboBoxModel)comboBox.getModel();
    comboBox = new JComboBox(model);
    final EnumValue value = property.getValue();
    for (int e = model.getSize()-1;  e >= 0;  --e) {
      if (((EnumValueLocalizer)model.getElementAt(e)).getBasicValue() == value) {
        comboBox.setSelectedIndex(e);
      }
    }
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  @SuppressWarnings("unchecked")
  public void save()
  {
    // note: combobox isn't generalized
    EnumValueLocalizer<E> value = (EnumValueLocalizer<E>)comboBox.getSelectedItem();
    property.setValue(value.getBasicValue());
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
  }
}
