// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.settings.EnumValue;

import javax.swing.*;
import java.util.Locale;

/**
 * Enum toggled property.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class EnumCheckedProxyProperty<E>
        extends AbstractBasicSimpleEditableChangeableItem
{
  private static final long serialVersionUID = -7736211554320924720L;

  @NotNull
  private final ButtonGroup buttonGroup;
  @NotNull
  private final SwingEnumProperty<E> enumProperty;
  @NotNull
  private final EnumValue<E> onValue;
  @NotNull
  private final EditableProperty switchedProperty;

  /**
   * Constructor.
   * @param basicName           basic name
   * @param baseTag             basic property tag
   * @param buttonGroup         button group for property
   * @param enumProperty        enum property
   * @param onValue             enum value for which the switched property is enabled
   * @param switchedProperty    switched property
   */
  public EnumCheckedProxyProperty(@NotNull String basicName, @NotNull String baseTag,
                                  @NotNull ButtonGroup buttonGroup,
                                  @NotNull SwingEnumProperty<E> enumProperty,
                                  @NotNull EnumValue<E> onValue,
                                  @NotNull EditableProperty switchedProperty)
  {
    super(basicName, baseTag);
    this.buttonGroup  = buttonGroup;
    this.enumProperty = enumProperty;
    this.onValue = onValue;
    this.switchedProperty = switchedProperty;
  }

  /**
   * Get the editor provider which provides editor components for editing this preference property.
   *
   * @param l locale used for i18n, {@code null} for default
   * @return editor provider
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return new EnumCheckedProxyEditorProvider<>(buttonGroup,
                                                 enumProperty,
                                                 onValue,
                                                 switchedProperty.getEditorProvider(l));
  }
}
