// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.util.settings.EnumValue;

import javax.swing.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

/**
 * A proxy editor provider which is toggled by an enu,.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <E> enum type
 */
public class EnumCheckedProxyEditorProvider<E>
        implements EditorProvider,
                   PropertyChangeListener
{
  @NotNull
  private final SwingEnumProperty<E> enumProperty;
  @NotNull
  private final EnumValue<E> onValue;
  @NotNull
  private final EditorProvider baseEditorProvider;
  /** Label component. */
  private final JRadioButton label;
  /** Editor component. */
  private final JComponent editor;
  /** Was this editor switched on initially. */
  private final boolean wasOn;

  /**
   * Constructor.
   * @param buttonGroup   button group for radio buttons
   * @param enumProperty  enum property holding the enum which toggles this proxy
   * @param onValue       enum value which toggles this proxy
   * @param baseEditorProvider  base editor provider
   */
  public EnumCheckedProxyEditorProvider(@NotNull ButtonGroup buttonGroup,
                                        @NotNull final SwingEnumProperty<E> enumProperty,
                                        @NotNull final EnumValue<E> onValue,
                                        @NotNull EditorProvider baseEditorProvider)
  {
    this.enumProperty = enumProperty;
    this.onValue = onValue;
    this.baseEditorProvider = baseEditorProvider;
    label = new JRadioButton(AbstractBasicEditorProvider.getLabelTextWithPunctuation(baseEditorProvider.getLabelText()));
    buttonGroup.add(label);
    editor = baseEditorProvider.getEditor();
    wasOn = enumProperty.getValue() == onValue;
    editor.setEnabled(wasOn);
    label.setSelected(wasOn);
    label.addItemListener(e -> {
      final boolean selected = label.isSelected();
      editor.setEnabled(selected);
      if (selected) {
        enumProperty.setValue(onValue);
      }
    });
    enumProperty.addValueChangeListenerWeakly(this);
  }

  /**
   * This method gets called when a bound property is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the property that has changed.
   */
  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    label.setSelected(evt.getNewValue() == onValue);
  }

  /**
   * Get the label text.
   *
   * @return label text
   */
  @Nullable
  @Override
  public String getLabelText()
  {
    return label.getText();
  }

  /**
   * Get a label component for this editor.
   * If the returned component is null the {@link #getEditor() editor component}
   * is inserted without a label.
   *
   * @return something used as a label component, or {@code null}
   */
  @Nullable
  @Override
  public JComponent getLabel()
  {
    return label;
  }

  /**
   * Get a title text for this editor component.
   *
   * @return a title text or {@code null} if a title makes no sense
   */
  @Nullable
  @Override
  public String getTitleText()
  {
    return baseEditorProvider.getTitleText();
  }

  /**
   * Get a tool tip string for this editor component.
   *
   * @return tooltip text or {@code null} if no tooltip is used
   */
  @Nullable
  @Override
  public String getToolTipText()
  {
    return baseEditorProvider.getToolTipText();
  }

  /**
   * Get a description for the property used.
   *
   * @return description, possibly multi-line
   */
  @Nullable
  @Override
  public String getDescription()
  {
    return baseEditorProvider.getDescription();
  }

  /**
   * Get an component for editing .
   *
   * @return editor component, use non-editing components like labels for fix properties
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return editor;
  }

  /**
   * Is the value in the editor component valid?
   *
   * @return {@code true} if the value is valid<br>
   * {@code false} otherwise
   */
  @Override
  public boolean isValidValue()
  {
    return baseEditorProvider.isValidValue();
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    baseEditorProvider.reset();
    label.setSelected(wasOn);
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    baseEditorProvider.save();
    if (label.isSelected()) {
      enumProperty.setValue(onValue);
    }
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
    // empty
  }
}
