// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Types;
import de.caff.i18n.I18n;

import java.beans.PropertyChangeListener;
import java.util.*;

/**
 * A group of editable properties.
 * @param <T> editable property type
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class EditablePropertyGroup<T extends EditableProperty>
        implements EditableProperty
{
  /** A collection of {@link EditableProperty} elements. */
  protected final Collection<T> propertyList = Types.synchronizedCollection(new LinkedList<T>());
  /** The common title for the collected preference properties. */
  protected final String nameTag;
  /** Are there tabs to create for each added property? */
  protected final boolean tabbed;
  /** The last tab position if tabbed. */
  private int tabIndex = 0;

  /**
   *  Create a preference group.
   *  @param nameTag the name tag
   *  @param tabbed  are there tabs to create for each added property?
   */
  public EditablePropertyGroup(@NotNull String nameTag, boolean tabbed)
  {
    this.nameTag = nameTag;
    this.tabbed  = tabbed;
  }

  /**
   *  Create a preference group.
   *  This group is not tabbed.
   *  @param nameTag the name tag
   *  @param properties preferences to collect in this group
   */
  @SafeVarargs
  public EditablePropertyGroup(@NotNull String nameTag, T... properties)
  {
    this(nameTag, false, properties);
  }

  /**
   *  Create a preference group.
   *  @param nameTag the name tag
   *  @param tabbed  are there tabs to create for each added property?
   *  @param properties preferences to collect in this group
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public EditablePropertyGroup(@NotNull String nameTag, boolean tabbed, T... properties)
  {
    this(nameTag, tabbed);
    if (properties != null) {
      propertyList.addAll(Types.asList(properties));
    }
  }

  /**
   *  Create a preference group.
   *  This group is not tabbed.
   *  @param nameTag the name tag
   *  @param properties preferences to collect in this group
   */
  public EditablePropertyGroup(@NotNull String nameTag, @NotNull Collection<? extends T> properties)
  {
    this(nameTag, false, properties);
  }

  /**
   *  Create a preference group.
   *  @param nameTag the name tag
   *  @param tabbed  are there tabs to create for each added property?
   *  @param properties preferences to collect in this group
   */
  public EditablePropertyGroup(@NotNull String nameTag, boolean tabbed, @NotNull Collection<? extends T> properties)
  {
    this(nameTag, tabbed);
    propertyList.addAll(properties);
  }

  /**
   * Get the editor component for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor component
   */
  @NotNull
  @Override
  public EditorProvider getEditorProvider(@Nullable Locale l)
  {
    return tabbed  ? new TabbedGroupEditor(this, l) :  new PanelGroupEditor(this, l);
  }

  /**
   * Get a common title for the editor components.
   * This will be displayed in a titled border around the components.
   * It's okay to return nothing especially if {@link #getEditorProvider(java.util.Locale)}
   * does return only 1 editor component.
   *
   * @param l locale used for i18n
   * @return common title or {@code null}
   */
  @Nullable
  @Override
  public String getName(@Nullable Locale l)
  {
    return I18n.getString(nameTag, l);
  }

  /**
   *  Get the i18n name tag.
   *  @return name tag
   */
  @NotNull
  String getNameTag()
  {
    return nameTag;
  }

  /**
   *  Get an iterator over the collected properties.
   *  @return property iterator
   */
  Iterator<T> propertyIterator()
  {
    return propertyList.iterator();
  }

  /**
   * Add a listener which will be called for all value changes.
   *
   * @param listener value change listener to add
   */
  @Override
  public void addValueChangeListener(@NotNull PropertyChangeListener listener)
  {
    for (EditableProperty property : propertyList) {
      property.addValueChangeListener(listener);
    }
  }

  /**
   * Remove a listener which will be called for all value changes.
   *
   * @param listener value change listener to remove
   */
  @Override
  public void removeValueChangeListener(@NotNull PropertyChangeListener listener)
  {
    for (EditableProperty property : propertyList) {
      property.removeValueChangeListener(listener);
    }
  }

  /**
   * Add a listener which will be called for all value changes.
   *
   * @param listener value change listener to add
   */
  @Override
  public void addValueChangeListenerWeakly(@NotNull PropertyChangeListener listener)
  {
    for (EditableProperty property : propertyList) {
      property.addValueChangeListenerWeakly(listener);
    }
  }

  /**
   * Remove a listener which will be called for all value changes.
   *
   * @param listener value change listener to remove
   */
  @Override
  public void removeValueChangeListenerWeakly(@NotNull PropertyChangeListener listener)
  {
    for (EditableProperty property : propertyList) {
      property.removeValueChangeListenerWeakly(listener);
    }
  }

  /**
   *  Add a property to the preference list.
   * <p>
   *  Note that listeners added before are not automatically added to the new property.
   *  @param property    property to add
   */
  protected void addProperty(@NotNull T property)
  {
    propertyList.add(property);
  }

  /**
   *  Get the tab pos.
   *  @return selected tab position
   */
  int getTabIndex()
  {
    return tabIndex;
  }

  /**
   *  Set the tab position.
   *  @param tabIndex tab position index
   */
  void setTabIndex(int tabIndex)
  {
    this.tabIndex = tabIndex;
  }
}
