// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;

import java.util.prefs.Preferences;

/**
 *  A group of editable preference properties.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class EditablePreferenceGroup
        extends EditablePropertyGroup<EditablePreferenceProperty>
        implements EditablePreferenceProperty
{
  /** The suffix for the tab index preference property. */
  public static final String TABBED_SUFFIX = "-TABIDX";

  /**
   *  Create a preference group.
   *  @param nameTag the name tag
   *  @param tabbed  are there tabs to create for each added property?
   */
  public EditablePreferenceGroup(@NotNull String nameTag, boolean tabbed)
  {
    super(nameTag, tabbed);
  }

  /**
   *  Create a preference group.
   *  @param nameTag the name tag
   *  @param tabbed  are there tabs to create for each added property?
   *  @param properties preferences to collect in this group
   */
  public EditablePreferenceGroup(@NotNull String nameTag, boolean tabbed, EditablePreferenceProperty... properties)
  {
    super(nameTag, tabbed, properties);
  }


  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  @Override
  public void readFrom(@NotNull Preferences preferences)
  {
    for (EditablePreferenceProperty property : propertyList) {
      property.readFrom(preferences);
    }
    if (tabbed) {
      setTabIndex(preferences.getInt(nameTag+TABBED_SUFFIX, getTabIndex()));
    }
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  @Override
  public void storeTo(@NotNull Preferences preferences)
  {
    for (EditablePreferenceProperty property : propertyList) {
      property.storeTo(preferences);
    }
    if (tabbed) {
      preferences.putInt(nameTag+TABBED_SUFFIX, getTabIndex());
    }
  }

}
