// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;

import javax.swing.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.util.Locale;

/**
 *  Editor provider for directory properties.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class DirectoryEditor
        extends AbstractBasicEditorProvider
        implements PropertyChangeListener
{
  /** The associated preference property. */
  private final SwingDirectoryPreferenceProperty property;
  /** Box with label and button. */
  private final Box box = Box.createHorizontalBox();
  /** Label displaying the current selection. */
  private final JLabel label;
  /** Button which starts a color chooser dialog. */
  private final JButton button;
  /** The locally changed color. */
  private File directory;

  /**
   * Constructor.
   * @param prop property to edit
   * @param l    locale to use
   */
  public DirectoryEditor(SwingDirectoryPreferenceProperty prop, final Locale l)
  {
    super(prop, l);
    this.property = prop;
    directory = property.getDirectory();
    label = new JLabel(directory.getPath());
    button = new JButton("...");
    button.addActionListener(e -> {
      JFileChooser chooser = new JFileChooser(directory);
      chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
      chooser.setDialogTitle(property.getName(l));
      if (chooser.showSaveDialog(button) == JFileChooser.APPROVE_OPTION) {
        setDirectory(chooser.getSelectedFile());
      }
    });
    box.add(label);
    box.add(Box.createHorizontalGlue());
    box.add(button);
    property.addValueChangeListenerWeakly(this);
  }

  private void setDirectory(File dir)
  {
    directory = dir;
    label.setText(dir == null  ?   ""  :  dir.getPath());
  }

  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return box;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    setDirectory(property.getDirectory());
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    property.setDirectory(directory);
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
    property.removeValueChangeListener(this);
  }

  /**
   * This method gets called when a bound property is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the property that has changed.
   */

  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    setDirectory((File)evt.getNewValue());
  }

}
