// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.i18n.swing.RJButton;

import javax.swing.*;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Locale;

/**
 *  Editor for colors.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class ColorEditor
        extends AbstractBasicEditorProvider
        implements PropertyChangeListener
{
  private static final int IMAGE_WIDTH  = 32;
  private static final int IMAGE_HEIGHT = 16;

  /** The associated preference property. */
  private final SwingColorProperty property;
  /** Button which starts a color chooser dialog. */
  private final JButton button;
  /** The locally changed color. */
  private Color color;

  /**
   * Constructor.
   * @param prop property to edit
   * @param l    locale to use
   */
  public ColorEditor(@NotNull SwingColorProperty prop, Locale l)
  {
    this(prop, l, false);
  }
  /**
   * Constructor.
   * @param prop property to edit
   * @param l    locale to use
   * @param autoSave save the changed color immediately?
   */
  public ColorEditor(@NotNull SwingColorProperty prop, final Locale l, final boolean autoSave)
  {
    super(prop, l);
    this.property = prop;
    button = new RJButton("btChangeColor");
    button.setLocale(l);
    button.addActionListener(e -> {
      final Color newColor = JColorChooser.showDialog(button, property.getName(l), color);
      if (newColor != null) {
        setColor(newColor);
        if (autoSave) {
          save();
        }
      }
    });
    property.addValueChangeListenerWeakly(this);

    setColor(prop.getValue());
  }

  private void setColor(Color color)
  {
    this.color = color;
    button.setIcon(createColoredIcon(color));
  }

  static Icon createColoredIcon(Color iconColor)
  {
    Image image = new BufferedImage(IMAGE_WIDTH, IMAGE_HEIGHT, BufferedImage.TYPE_INT_RGB);
    Graphics g = image.getGraphics();
    g.setColor(iconColor);
    g.fillRect(0, 0, IMAGE_WIDTH, IMAGE_HEIGHT);
    g.dispose();
    return new ImageIcon(image);
  }

  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return button;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    setColor(property.getValue());
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    property.setValue(color);
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
    property.removeValueChangeListener(this);
  }

  /**
   * This method gets called when a bound property is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the property that has changed.
   */

  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    if (color.equals(evt.getOldValue())) {
      setColor((Color)evt.getNewValue());
    }
  }
}
