// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Locale;

/**
 *  Editor provider for boolean properties.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
class BooleanEditor
        extends AbstractBasicEditorProvider
        implements PropertyChangeListener
{
  /** The property to edit. */
  private final SwingBooleanProperty property;
  /** The checkbox representing the property. */
  private final JCheckBox checkBox;

  /**
   * Constructor.
   * @param property property to edit
   * @param l        locale to use
   */
  public BooleanEditor(@NotNull SwingBooleanProperty property,
                       @Nullable Locale l)
  {
    super(property, l);
    this.property = property;
    checkBox = new JCheckBox((String)null, property.getValue());
    if (property.isAutoSave()) {
      checkBox.addChangeListener(e -> save());
    }
    property.addValueChangeListenerWeakly(this);
  }

  /**
   * Get an component for editing .
   *
   * @return editor component
   */
  @NotNull
  @Override
  public JComponent getEditor()
  {
    return checkBox;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    checkBox.setSelected(property.getValue());
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    property.setValue(checkBox.isSelected());
  }

  /**
   * This method gets called when a bound property is changed.
   *
   * @param evt A PropertyChangeEvent object describing the event source
   *            and the property that has changed.
   */
  @Override
  public void propertyChange(PropertyChangeEvent evt)
  {
    boolean newValue = ((Boolean)evt.getNewValue()).booleanValue();
    if (newValue != checkBox.isSelected()) {
      checkBox.setSelected(newValue);
    }
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
    property.removeValueChangeListener(this);
  }
}
