// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.Nullable;

import javax.swing.*;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Locale;

/**
 *  Base class for editors which handle a group of preference properties.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
abstract class AbstractBasicGroupEditor
        extends AbstractBasicEditorProvider
{
  /** The editor components. */
  private final Collection<EditorProvider> editorProviders = new LinkedList<>();

  /**
   *  Constructor.
   *  @param baseTag basic i18n tag
   *  @param l       locale
   */
  public AbstractBasicGroupEditor(String baseTag, Locale l)
  {
    super(baseTag, l);
  }

  /**
   * Get a label component for this editor.
   * If the returned component is null the {@link #getEditor() editor component}
   * is inserted without a label.
   *
   * @return something used as a label component, or {@code null}
   */
  @Nullable
  @Override
  public JComponent getLabel()
  {
    return null;
  }

  /**
   * Get a title text for this editor component.
   *
   * @return a title text or {@code null} if a title makes no sense
   */
  @Override public String getTitleText()
  {
    return super.getLabelText();
  }

  /**
   *  Is the value in the editor component valid?
   *  @return the answer
   */
  @Override public boolean isValidValue()
  {
    for (EditorProvider provider: editorProviders) {
      if (!provider.isValidValue()) {
        return false;
      }
    }
    return true;
  }

  /**
   * Reset the value in the editor to the basic value.
   */
  @Override
  public void reset()
  {
    for (EditorProvider provider: editorProviders) {
      provider.reset();
    }
  }

  /**
   * Set the basic value from the editor.
   */
  @Override
  public void save()
  {
    for (EditorProvider provider: editorProviders) {
      provider.save();
    }
  }

  /**
   * Called when the editor provider is no longer used.
   */
  @Override
  public void goodBye()
  {
    for (EditorProvider provider: editorProviders) {
      provider.goodBye();
    }
  }

  /**
   *  Add an editor provider.
   *  @param provider provider to add
   */
  protected void addEditorProvider(EditorProvider provider)
  {
    editorProviders.add(provider);
  }
}

