// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2005-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.util.settings.SettingsTools;

import javax.swing.*;
import java.util.Locale;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
abstract class AbstractBasicEditorProvider
        implements EditorProvider
{
  static {
    I18n.addAppResourceBase("de.caff.util.settings.swing.SettingsResourceBundle");
  }
  /** The name of the label. */
  private final String labelName;
  /** Tool tip text to use. */
  private final String toolTipText;
  /** Description to use. */
  private final String description;

  /**
   * Constructor.
   * @param baseTag base tag (will be expanded by various suffixes)
   * @param l       locale
   */
  protected AbstractBasicEditorProvider(@NotNull String baseTag, @Nullable Locale l)
  {
    labelName = SettingsTools.getOptionalI18nString(baseTag+I18n.SUFFIX_TEXT, l);
    toolTipText = SettingsTools.getOptionalI18nString(baseTag+ I18n.SUFFIX_TOOLTIP, l);
    description = SettingsTools.getOptionalI18nString(baseTag+I18n.SUFFIX_DESCRIPTION, l);
  }

  /**
   * Constructor.
   * @param property property to edit
   * @param l        locale to use
   */
  protected AbstractBasicEditorProvider(@NotNull AbstractBasicSimpleEditableChangeableItem property,
                                        @Nullable Locale l)
  {
    this(property.getBaseTag(), l);
  }

  /**
   *  Is the value in the editor component valid?
   *  @return the answer
   */
  @Override
  public boolean isValidValue()
  {
    return true;
  }

  /**
   * Get the label text.
   * @return label text
   */
  @Override
  @Nullable
  public String getLabelText()
  {
    return labelName;
  }

  /**
   * Get the label text including potential punctuation.
   * @param text basic label text
   * @return label text including punctuation,
   *         or {@code  null} if there is no label text defined
   */
  @Nullable
  public static String getLabelTextWithPunctuation(@Nullable String text)
  {
    if (text == null) {
      return null;
    }
    String punctuation = I18n.getString("valueNamePunctuation");
    boolean endsWithPunctuation =
            punctuation.indexOf(text.charAt(text.length() - 1)) >= 0;
    return I18n.format(endsWithPunctuation
                               ? "valueNameFormatPunctuation"
                               : "valueNameFormat",
                       text,
                       text.substring(0, text.length() - 1));
  }

  /**
   * Get a label component for this editor.
   * If the returned component is null the {@link #getEditor() editor component}
   * is inserted without a label.
   *
   * @return something used as a label component, or {@code null}
   */
  @Nullable
  @Override
  public JComponent getLabel()
  {
    if (labelName == null) {
      return null;
    }
    return new JLabel(getLabelTextWithPunctuation(labelName));
  }

  /**
   * Get a title text for this editor component.
   *
   * @return a title text or {@code null} if a title makes no sense
   */
  @Nullable
  @Override
  public String getTitleText()
  {
    return null;
  }

  /**
   * Get a tool tip string for this editor component.
   *
   * @return tooltip text or {@code null} if no tooltip is used
   */
  @Nullable
  @Override
  public String getToolTipText()
  {
    return toolTipText;
  }

  /**
   * Get a description for the property used.
   *
   * @return description, possibly multi-line
   */
  @Nullable
  @Override
  public String getDescription()
  {
    return description;
  }

}
