// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2010-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.i18n.swing;

import de.caff.annotation.NotNull;
import de.caff.generics.Empty;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.util.Utility;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;
import java.util.MissingResourceException;

/**
 * Tabbed pane which might react to locale changes.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJTabbedPane
        extends JTabbedPane
        implements Localizable
{
  private static final long serialVersionUID = -2406799170208381516L;
  /** Resource tags for the tabs. {@code null} if tab doesn't have a resource. */
  private String[] baseTags = Empty.STRING_ARRAY;

  /**
   * Common initializations.
   */
  private void init()
  {
    super.setLocale(I18n.getDefaultLocale());
  }

  /**
   * Creates an empty {@code TabbedPane} with a default
   * tab placement of {@code JTabbedPane.TOP}.
   *
   * @see #addTab
   */
  public RJTabbedPane()
  {
    super.setLocale(I18n.getDefaultLocale());
    init();
  }

  /**
   * Creates an empty {@code TabbedPane} with the specified tab placement
   * of either: {@code JTabbedPane.TOP}, {@code JTabbedPane.BOTTOM},
   * {@code JTabbedPane.LEFT}, or {@code JTabbedPane.RIGHT}.
   *
   * @param tabPlacement the placement for the tabs relative to the content
   * @see #addTab
   */
  public RJTabbedPane(int tabPlacement)
  {
    super(tabPlacement);
    init();
  }

  /**
   * Creates an empty {@code TabbedPane} with the specified tab placement
   * and tab layout policy.  Tab placement may be either:
   * {@code JTabbedPane.TOP}, {@code JTabbedPane.BOTTOM},
   * {@code JTabbedPane.LEFT}, or {@code JTabbedPane.RIGHT}.
   * Tab layout policy may be either: {@code JTabbedPane.WRAP_TAB_LAYOUT}
   * or {@code JTabbedPane.SCROLL_TAB_LAYOUT}.
   *
   * @param tabPlacement    the placement for the tabs relative to the content
   * @param tabLayoutPolicy the policy for laying out tabs when all tabs will not fit on one run
   * @throws IllegalArgumentException if tab placement or tab layout policy are not
   *                                  one of the above supported values
   * @see #addTab
   * @since 1.4
   */
  public RJTabbedPane(int tabPlacement, int tabLayoutPolicy)
  {
    super(tabPlacement, tabLayoutPolicy);
    init();
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   *
   * @see #registerKeyboardAction
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Notifies this component that it no longer has a parent component.
   * When this method is invoked, any {@code KeyboardAction}s
   * set up in the the chain of parent components are removed.
   *
   * @see #registerKeyboardAction
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   * Adds a {@code component} represented by a {@code title}
   * and no icon.
   * Cover method for {@code insertTab}.
   *
   * @param baseTag     the title to be displayed in this tab
   * @param component the component to be displayed when this tab is clicked
   * @see #insertTab
   * @see #removeTabAt
   */
  public void addResourcedTab(@NotNull String baseTag,
                              @NotNull Component component)
  {
    super.addTab(I18n.getString(baseTag + I18n.SUFFIX_TEXT), component);
    updateBaseTagSize();
    int tabIndex = getTabCount() - 1;
    baseTags[tabIndex] = baseTag;
    resetResources(tabIndex);
  }

  private void updateBaseTagSize()
  {
    String[] tags = new String[getTabCount()];
    System.arraycopy(baseTags, 0, tags, 0, Math.min(tags.length, baseTags.length));
    baseTags = tags;
  }

  /**
   * Reset the appearance of a tab according to the locale of this component.
   * @param tabIndex tab index
   */
  private void resetResources(int tabIndex)
  {
    resetResources(tabIndex, getLocale());
  }

  /**
   * Reset the appearance of a tab according to a given locale.
   * @param tabIndex tab index
   * @param locale   locale
   */
  private void resetResources(int tabIndex, Locale locale)
  {
    String baseTag = baseTags[tabIndex];
    if (baseTag != null) {
      super.setTitleAt(tabIndex, I18n.getString(baseTag + I18n.SUFFIX_TEXT, locale));
      try {
        setToolTipTextAt(tabIndex, I18n.getString(baseTag+I18n.SUFFIX_TOOLTIP, locale));
      } catch (MissingResourceException e) {
      }
      try {
        setMnemonicAt(tabIndex, I18n.getString(baseTag+I18n.SUFFIX_MNEMONIC, locale).charAt(0));
      } catch (Exception e) {
      }
      try {
        Image image = Utility.loadImage(I18n.getString(baseTag+I18n.SUFFIX_DISABLED_ICON, locale));
        if (image != null) {
          setDisabledIconAt(tabIndex, new ImageIcon(image));
        }
        else {
          setDisabledIconAt(tabIndex, null);
        }
      } catch (MissingResourceException e) {
        setDisabledIconAt(tabIndex, null);
      }
      try {
        Image image = Utility.loadImage(I18n.getString(baseTag+I18n.SUFFIX_ICON, locale));
        if (image != null) {
          setIconAt(tabIndex, new ImageIcon(image));
        }
        else {
          setIconAt(tabIndex, null);
        }
      } catch (MissingResourceException e) {
        setIconAt(tabIndex, null);
      }
    }
  }

  /**
   * Removes all the tabs and their corresponding components
   * from the {@code tabbedpane}.
   *
   * @see #addTab
   * @see #removeTabAt
   */
  @Override
  public void removeAll()
  {
    super.removeAll();
    updateBaseTagSize();
  }

  /**
   * Removes the tab and component which corresponds to the specified index.
   *
   * @param index the index of the component to remove from the
   *              {@code tabbedpane}
   * @throws IndexOutOfBoundsException if index is out of range
   *                                   (index &lt; 0 || index &gt;= tab count)
   * @see #addTab
   * @see #removeTabAt
   */
  @Override
  public void remove(int index)
  {
    super.remove(index);
  }

  /**
   * Removes the tab at {@code index}.
   * After the component associated with {@code index} is removed,
   * its visibility is reset to true to ensure it will be visible
   * if added to other containers.
   *
   * @param index the index of the tab to be removed
   * @throws IndexOutOfBoundsException if index is out of range
   *                                   (index &lt; 0 || index &gt;= tab count)
   * @see #addTab
   * @see #insertTab
   */
  @Override
  public void removeTabAt(int index)
  {
    super.removeTabAt(index);
    System.arraycopy(baseTags, index + 1, baseTags, index, getTabCount() - index);
    updateBaseTagSize();
  }

  /**
   * Removes the specified {@code Component} from the
   * {@code JTabbedPane}.
   *
   * @param component the component to remove from the tabbedpane
   * @throws NullPointerException if {@code component} is null.
   * @see #addTab
   * @see #removeTabAt
   */
  @Override
  public void remove(Component component)
  {
    super.remove(component);
  }

  /**
   * Sets the title at {@code index} to {@code title} which
   * can be {@code null}.
   * An internal exception is raised if there is no tab at that index.
   *
   * @param index the tab index where the title should be set
   * @param title the title to be displayed in the tab
   * @throws IndexOutOfBoundsException if index is out of range
   *                                   (index &lt; 0 || index &gt;= tab count)
   * attribute: visualUpdate true
   * description: The title at the specified tab index.
   * @see #getTitleAt
   */
  @Override
  public void setTitleAt(int index, String title)
  {
    super.setTitleAt(index, title);
    baseTags[index] = null;
  }

  /**
   * Sets the tooltip text at {@code index} to {@code toolTipText}
   * which can be {@code null}.
   * An internal exception is raised if there is no tab at that index.
   *
   * @param index       the tab index where the tooltip text should be set
   * @param toolTipText the tooltip text to be displayed for the tab
   * @throws IndexOutOfBoundsException if index is out of range
   *                                   (index &lt; 0 || index &gt;= tab count)
   * description: The tooltip text at the specified tab index.
   * @see #getToolTipTextAt
   */
  @Override
  public void setToolTipTextAt(int index, String toolTipText)
  {
    super.setToolTipTextAt(index, toolTipText);
    baseTags[index] = null;
  }

  /**
   * Sets the disabled icon at {@code index} to {@code icon}
   * which can be {@code null}.
   * An internal exception is raised if there is no tab at that index.
   *
   * @param index        the tab index where the disabled icon should be set
   * @param disabledIcon the icon to be displayed in the tab when disabled
   * @throws IndexOutOfBoundsException if index is out of range
   *                                   (index &lt; 0 || index &gt;= tab count)
   * attribute: visualUpdate true
   * description: The disabled icon at the specified tab index.
   * @see #getDisabledIconAt
   */
  @Override
  public void setDisabledIconAt(int index, Icon disabledIcon)
  {
    super.setDisabledIconAt(index, disabledIcon);
    baseTags[index] = null;
  }

  /**
   * Sets the icon at {@code index} to {@code icon} which can be
   * {@code null}. Does not set disabled icon at {@code icon}
   * To set disabled icon, use {@code setDisableIconAt()}.
   * An internal exception is raised if there is no tab at that index.
   *
   * @param index the tab index where the icon should be set
   * @param icon  the icon to be displayed in the tab
   * @throws IndexOutOfBoundsException if index is out of range
   *                                   (index &lt; 0 || index &gt;= tab count)
   * attribute: visualUpdate true
   * description: The icon at the specified tab index.
   * @see #setDisabledIconAt
   * @see #getIconAt
   */
  @Override
  public void setIconAt(int index, Icon icon)
  {
    super.setIconAt(index, icon);
    baseTags[index] = null;
  }

  /**
   * Sets the locale of this component.  This is a bound property.
   *
   * @param l the locale to become this component's locale
   * @see #getLocale
   * @since JDK1.1
   */
  @Override
  public void setLocale(Locale l)
  {
    super.setLocale(l);
    for (int t = getTabCount() - 1;  t >= 0;  --t) {
      resetResources(t, l);
    }
  }
}
