// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2003-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.i18n.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;
import de.caff.util.settings.IntegerPreferenceProperty;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Locale;

/**
 *  JSplitPane which notifies it's components of locale changes.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJSplitPane
        extends JSplitPane
        implements Localizable
{

  private static final long serialVersionUID = -3904535612520147699L;

  public RJSplitPane()
  {
    super.setLocale(I18n.getDefaultLocale());
  }

  public RJSplitPane(int newOrientation)
  {
    super(newOrientation);
    super.setLocale(I18n.getDefaultLocale());
  }

  public RJSplitPane(int newOrientation, boolean newContinuousLayout)
  {
    super(newOrientation, newContinuousLayout);
    super.setLocale(I18n.getDefaultLocale());
  }

  public RJSplitPane(int newOrientation, Component newLeftComponent, Component newRightComponent)
  {
    super(newOrientation, newLeftComponent, newRightComponent);
    super.setLocale(I18n.getDefaultLocale());
  }

  public RJSplitPane(int newOrientation, boolean newContinuousLayout, Component newLeftComponent, Component newRightComponent)
  {
    super(newOrientation, newContinuousLayout, newLeftComponent, newRightComponent);
    super.setLocale(I18n.getDefaultLocale());
  }

  public RJSplitPane(Locale l)
  {
    setLocale(l);
  }

  public RJSplitPane(int newOrientation, Locale l)
  {
    super(newOrientation);
    setLocale(l);
  }

  public RJSplitPane(int newOrientation, boolean newContinuousLayout, Locale l)
  {
    super(newOrientation, newContinuousLayout);
    setLocale(l);
  }

  public RJSplitPane(int newOrientation, Component newLeftComponent, Component newRightComponent, Locale l)
  {
    super(newOrientation, newLeftComponent, newRightComponent);
    setLocale(l);
  }

  public RJSplitPane(int newOrientation, boolean newContinuousLayout, Component newLeftComponent, Component newRightComponent, Locale l)
  {
    super(newOrientation, newContinuousLayout, newLeftComponent, newRightComponent);
    setLocale(l);
  }

  /**
   * Setup usage of a preference property for keeping the divider's position.
   * This should only be called once at program start.
   * @param dividerPositionPreferenceProperty preference property used for the divider position
   * @param defaultSplitPosition              default position of the divider if preference is giving back 0 or less
   */
  public void setupPreferences(@Nullable final IntegerPreferenceProperty dividerPositionPreferenceProperty,
                               final int defaultSplitPosition)
  {
    if (dividerPositionPreferenceProperty == null) {
      return;
    }
    int splitPos = dividerPositionPreferenceProperty.getValue();
    if (splitPos > 0) {
      setDividerLocation(splitPos);
    }
    else {
      addComponentListener(new ComponentAdapter()
      {
        @Override
        public void componentResized(ComponentEvent e)
        {
          // if component has a reasonable size set location and remove this listener
          if (e.getComponent().getWidth() > 32 || e.getComponent().getHeight() > 32) {
            setDividerLocation(Math.min(getMaximumDividerLocation(), Math.max(getMinimumDividerLocation(), defaultSplitPosition)));
            e.getComponent().removeComponentListener(this);
          }
        }
      });
    }
    addPropertyChangeListener(DIVIDER_LOCATION_PROPERTY,
                              evt -> dividerPositionPreferenceProperty.setValue((Integer)evt.getNewValue()));
  }

  /**
   * Setup usage of a preference property for keeping the divider's position.
   * This should only be called once at program start.
   * @param dividerPositionPreferenceProperty preference property used for the divider position
   * @param defaultSplitPosition              default position of the divider if preference is giving back 0 or less
   */
  public void setupPreferences(@NotNull final IntegerPreferenceProperty dividerPositionPreferenceProperty,
                               final double defaultSplitPosition)
  {
    int splitPos = dividerPositionPreferenceProperty.getValue();
    if (splitPos > 0) {
      setDividerLocation(splitPos);
    }
    else {
      addComponentListener(new ComponentAdapter()
      {
        @Override
        public void componentResized(ComponentEvent e)
        {
          int size = getOrientation() == HORIZONTAL_SPLIT
                  ? getWidth()
                  : getHeight();
          // if component has a reasonable size set location and remove this listener
          if (size > 32) {
            int pos = (int)(defaultSplitPosition * size);
            setDividerLocation(Math.min(getMaximumDividerLocation(), Math.max(getMinimumDividerLocation(), pos)));
            e.getComponent().removeComponentListener(this);
          }
        }
      });
    }
    addPropertyChangeListener(DIVIDER_LOCATION_PROPERTY,
                              evt -> dividerPositionPreferenceProperty.setValue((Integer)evt.getNewValue()));
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Change the locale.
   *  @param  l   new locale
   */
  @Override
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      invalidate();
    }
  }

  /**
   *  Get the current locale. Don't throws {@code IllegalComponentStateException}.
   *  @return current locale or null, if unknown
   */
  @Override
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

}
