// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2016-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.i18n.swing;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.util.Locale;

/**
 * Popup menu which can forward resource changes and knows of
 * resizable icons.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 02, 2017
 */
public class RJPopupMenu
        extends JPopupMenu
        implements Localizable
{
  private static final long serialVersionUID = -147122873835078384L;

  private String baseTag;

  /**
   * Constructor.
   * This creates a menu without title.
   */
  public RJPopupMenu()
  {
  }

  /**
   * Constructor.
   * This creates a menu with a localizable title.
   * @param baseTag base tag
   */
  public RJPopupMenu(@NotNull String baseTag)
  {
    super(I18n.getString(baseTag + I18n.SUFFIX_TEXT));
    this.baseTag = baseTag;
  }

  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   * Set the basic resource tag.
   * @param tag resource tag
   */
  public void setTag(@Nullable String tag)
  {
    baseTag = tag;
    if (tag == null) {
      setName(null);
    }
    else {
      setName(I18n.getString(tag + I18n.SUFFIX_TEXT));
    }
  }

  @Override
  public void setLocale(Locale l)
  {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (l != null) {
      if (!l.equals(oldLocale)) {
        setName(I18n.getString(baseTag + I18n.SUFFIX_TEXT));
      }
    }
    else {
      setName(null);
    }
  }
}
