// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n.swing;

import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  JPanel which notifies it's components of locale changes.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJPanel 
  extends    JPanel 
  implements Localizable 
{
  private static final long serialVersionUID = -6547451295462119709L;

  /**
   *  Constructor.
   */
  public RJPanel() {
    super.setLocale(I18n.getDefaultLocale());
  }

  /**
   *  Constructor.
   *  @param isDoubleBuffered  double buffer this JPanel?
   */
  public RJPanel(boolean isDoubleBuffered) {
    super(isDoubleBuffered);
    super.setLocale(I18n.getDefaultLocale());
  }

  /**
   *  Constructor.
   *  @param  mgr  layout manager
   */
  public RJPanel(LayoutManager mgr) {
    super(mgr);
    super.setLocale(I18n.getDefaultLocale());
  }

  /**
   *  Constructor.
   *  @param  mgr  layout manager
   *  @param isDoubleBuffered  double buffer this JPanel?
   */
  public RJPanel(LayoutManager mgr, boolean isDoubleBuffered) {
    super(mgr, isDoubleBuffered);
    super.setLocale(I18n.getDefaultLocale());
  }

  /**
   *  Constructor.
   *  @param  isDoubleBuffered is this panel double-buffered?
   *  @param  l         locale 
   */
  public RJPanel(boolean isDoubleBuffered, Locale l) {
    this(isDoubleBuffered);
    setLocale(l);
  }

  /**
   *  Constructor.
   *  @param  mgr  layout manager
   *  @param  l         locale 
   */
  public RJPanel(LayoutManager mgr, Locale l) {
    super(mgr);
    setLocale(l);
  }

  /**
   *  Constructor.
   *  @param  mgr  layout manager
   *  @param isDoubleBuffered  double buffer this JPanel?
   *  @param  l         locale 
   */
  public RJPanel(LayoutManager mgr, boolean isDoubleBuffered, Locale l) {
    this(mgr, isDoubleBuffered);
    setLocale(l);
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Change the locale.
   *  @param  l   new locale
   */
  @Override
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      invalidate();
    }
  }

  /**
   *  Get the current locale. Don't throws {@code IllegalComponentStateException}.
   *  @return current locale or null, if unknown
   */
  @Override
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

}
