// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2002-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.i18n.swing;

import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  MenuItem with localized label.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class RJMenuItem 
  extends    JMenuItem 
  implements Localizable 
{
  private static final long serialVersionUID = -4552009139480307014L;
  protected String resTag;         // name of the resource of this menu's name

  /**
   *  @param  tag       resource tag for the item
   */
  public RJMenuItem(String tag) {
    super();
    super.setLocale(I18n.getDefaultLocale());
    setTag(tag);
  }

  /**
   *  @param  tag       resource tag for the item
   *  @param  l         locale
   */
  public RJMenuItem(String tag, Locale l) {
    super();
    setTag(tag, l);
  }

  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with {@code KeyboardAction} event listeners.
   */
  @Override
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides {@code JComponent.removeNotify} to check if
   * this button is currently set as the default button on the
   * {@code RootPane}, and if so, sets the {@code RootPane}'s
   * default button to {@code null} to ensure the
   * {@code RootPane} doesn't hold onto an invalid button reference.
   */
  @Override
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }

  /**
   *  Change the tag of this item.
   *  @param  tag       new resource tag 
   */
  public void setTag(String tag) {
    setTag(tag, getLocale());
  }

  /**
   *  Change the tag of this item.
   *  @param  tag       new resource tag
   *  @param  l         locale to use
   */
  public void setTag(String tag, Locale l) {
    resTag = tag;
    setLocale(l);
    I18nHelper.setProperties(this, resTag, l);
  }

  /**
   *  Change the locale.
   *  @param  l    new locale
   */
  @Override
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (resTag != null) {
        I18nHelper.setProperties(this, resTag, l);
      }
    }
  }

  /**
   *  Get the locale. Instead of throwing an {@code IllegalComponentStateException}
   *  when there's no locale we just return null.
   *  @return the Locale of this Label or null
   */
  @Override
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }
}

